/*
  avr8_ac.h - AC Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_ac_dfp.h"
#include "avr8_vref.h"

#if   defined(AC2)
#define AC_MAX_CH 3
#elif defined(AC1)
#define AC_MAX_CH 2
#elif defined(AC0)
#define AC_MAX_CH 1
#else
#define AC_MAX_CH 0
#endif

#if defined(AC_CMP_bm) && !defined(AC_CMPIF_bm)
  #define AC_CMPIF_bm AC_CMP_bm
#endif

class Ac
{
  public:

    /* Positive Input MUX Selection select */
    typedef enum
    {
#if defined(AC_MUXPOS_AINP0_gc)
      MUXPOS_AINP0 = AC_MUXPOS_AINP0_gc,  /* Positive Pin 0 */
#endif
#if defined(AC_MUXPOS_AINP1_gc)
      MUXPOS_AINP1 = AC_MUXPOS_AINP1_gc,  /* Positive Pin 1 */
#endif
#if defined(AC_MUXPOS_AINP2_gc)
      MUXPOS_AINP2 = AC_MUXPOS_AINP2_gc,  /* Positive Pin 2 */
#endif
#if defined(AC_MUXPOS_AINP3_gc)
      MUXPOS_AINP3 = AC_MUXPOS_AINP3_gc,  /* Positive Pin 3 */
#endif
#if defined(AC_MUXPOS_AINP4_gc)
      MUXPOS_AINP4 = AC_MUXPOS_AINP4_gc,  /* Positive Pin 4 */
#endif
#if defined(AC_MUXPOS_AINP5_gc)
      MUXPOS_AINP5 = AC_MUXPOS_AINP5_gc,  /* Positive Pin 5 */
      MUXPOS_AINP6 = AC_MUXPOS_AINP6_gc,  /* Positive Pin 6 */
#endif
    } MUXPOS;

    /* Negative Input MUX Selection select */
    typedef enum
    {
#if defined(AC_MUXNEG_AINN0_gc)
      MUXNEG_AINN0  = AC_MUXNEG_AINN0_gc,   /* Negative Pin 0 */
#endif
#if defined(AC_MUXNEG_AINN1_gc)
      MUXNEG_AINN1  = AC_MUXNEG_AINN1_gc,   /* Negative Pin 1 */
#endif
#if defined(AC_MUXNEG_AINN2_gc)
      MUXNEG_AINN2  = AC_MUXNEG_AINN2_gc,   /* Negative Pin 2 */
#endif
#if defined(AC_MUXNEG_AINN3_gc)
      MUXNEG_AINN3  = AC_MUXNEG_AINN3_gc,   /* Negative Pin 3 */
#endif
#if defined(AC_MUXNEG_VREF_gc)
      MUXNEG_VREF   = AC_MUXNEG_VREF_gc,    /* Voltage Reference */
#endif
#if defined(AC_MUXNEG_DACREF_gc)
      MUXNEG_DACREF = AC_MUXNEG_DACREF_gc,  /* DAC Voltage Reference */
#endif
#if defined(AC_MUXNEG_DAC_gc)
      MUXNEG_DAC    = AC_MUXNEG_DAC_gc,     /* DAC output */
#endif
    } MUXNEG;

    /* Hysteresis Mode select */
    typedef enum
    {
      HYSMODE_NONE   = AC_HYSMODE_NONE_gc,    /* No hysteresis */
      HYSMODE_SMALL  = AC_HYSMODE_SMALL_gc,   /* Small hysteresis */
      HYSMODE_MEDIUM = AC_HYSMODE_MEDIUM_gc,  /* Medium hysteresis */
      HYSMODE_LARGE  = AC_HYSMODE_LARGE_gc,   /* Large hysteresis */
    } HYSMODE;

    /* Power profile select */
    typedef enum
    {
#if defined(AC_POWER_PROFILE0_gc)
      PWMODE_PROFILE0 = AC_POWER_PROFILE0_gc,  /* Power profile 0, Shortest response time, highest consumption */
      PWMODE_PROFILE1 = AC_POWER_PROFILE1_gc,  /* Power profile 1 */
  #if defined(AC_POWER_PROFILE2_gc)
      PWMODE_PROFILE2 = AC_POWER_PROFILE2_gc,  /* Power profile 2 */
  #endif
  #if defined(AC_POWER_PROFILE3_gc)
      PWMODE_PROFILE3 = AC_POWER_PROFILE3_gc,  /* Power profile 3 */
  #endif
#else
      PWMODE_PROFILE0 = 0,
  #if defined(AC_LPMODE_bm)
      PWMODE_PROFILE1 = AC_LPMODE_bm,          /* Low Power Mode bit mask. */
  #endif
#endif
    } PWMODE;

    /* Interrupt Mode select */
    typedef enum
    {
#if defined(AC_INTMODE_NORMAL_BOTHEDGE_gc)
      INTMODE_BOTHEDGE       = AC_INTMODE_NORMAL_BOTHEDGE_gc,  /* Positive and negative inputs crosses */
      INTMODE_NEGEDGE        = AC_INTMODE_NORMAL_NEGEDGE_gc,   /* Positive input goes below negative input */
      INTMODE_POSEDGE        = AC_INTMODE_NORMAL_POSEDGE_gc,   /* Positive input goes above negative input */
#endif
#if defined(AC_INTMODE_WINDOW_ABOVE_gc)
      INTMODE_WINDOW_ABOVE   = AC_INTMODE_WINDOW_ABOVE_gc,     /* Window interrupt when input above both references */
      INTMODE_WINDOW_INSIDE  = AC_INTMODE_WINDOW_INSIDE_gc,    /* Window interrupt when input betweeen references */
      INTMODE_WINDOW_BELOW   = AC_INTMODE_WINDOW_BELOW_gc,     /* Window interrupt when input below both references */
      INTMODE_WINDOW_OUTSIDE = AC_INTMODE_WINDOW_OUTSIDE_gc,   /* Window interrupt when input outside reference */
#endif
    } INTMODE;

#if defined(AC_WINSEL_gm)
    /* Window selection mode */
    typedef enum AC_WINSEL_enum
    {
      WINSEL_DISABLED = AC_WINSEL_DISABLED_gc,  /* Window function disabled */
      WINSEL_UPSEL1   = AC_WINSEL_UPSEL1_gc,    /* Select ACn+1 as upper limit in window compare */
  #if defined(AC_WINSEL_UPSEL2_gc)
      WINSEL_UPSEL2   = AC_WINSEL_UPSEL2_gc,    /* Select ACn+2 as upper limit in window compare */
  #endif
    } WINSEL;

    /* Analog Comparator Window State select */
    typedef enum AC_WINSTATE_enum
    {
      WINSTATE_ABOVE   = AC_WINSTATE_ABOVE_gc,   /* Above window */
      WINSTATE_INSIDE  = AC_WINSTATE_INSIDE_gc,  /* Inside window */
      WINSTATE_BELOW   = AC_WINSTATE_BELOW_gc,   /* Below window */
    } WINSTATE;
#endif

    /* AC.STATUS  bit masks and bit positions */
    typedef enum
    {
      STATUS_CMPIF    = AC_CMPIF_bm,     /* Interrupt Enable bit mask. */
      STATUS_CMPSTATE = AC_CMPSTATE_bm,  /* Analog Comparator State bit mask. */
#if defined(AC_WINSTATE_gm)
      STATUS_WINSTATE = AC_WINSTATE_gm,  /* Analog Comparator Window State group mask. */
#endif
    } STATUS;

    /* Type of Callback Function */
    typedef void (*callback_t)(STATUS status);

    Ac(AC_t& ac) : _ac(&ac), _ch((&ac - &AC0) / sizeof(ac)), _callback(0)
    {
      _instances[_ch] = this;
    }

    /* virtual */ ~Ac(void)
    {
      end();
      _instances[_ch] = 0;
    }

    void begin(MUXPOS muxpos, MUXNEG muxneg, HYSMODE hysmode = HYSMODE_NONE, bool invert = false)
    {
      end();
#if defined(AC0_MUXCTRLA)
      _ac->MUXCTRLA = (invert ? AC_INVERT_bm : 0) | muxpos | muxneg;
#else
      _ac->MUXCTRL  = (invert ? AC_INVERT_bm : 0) | muxpos | muxneg;
#endif
#if defined(AC_DACREF_gm)
      dacref();
#endif
#if defined(AC_WINSEL_gm)
      winsel();
#endif
      _ac->INTCTRL  = 0;
      _ac->STATUS   = AC_CMPIF_bm;
      _ac->CTRLA    = hysmode | AC_ENABLE_bm;
    }

    void end(void)
    {
      _ac->CTRLA = 0;
    }

    void standby(bool enable = true, PWMODE pwmode = PWMODE_PROFILE0)
    {
#if defined(AC_POWER_gm)
      _ac->CTRLA = (_ac->CTRLA & ~(AC_RUNSTDBY_bm | AC_POWER_gm)) | (enable ? AC_RUNSTDBY_bm : 0) | pwmode;
#elif defined(AC_LPMODE_bm)
      _ac->CTRLA = (_ac->CTRLA & ~(AC_RUNSTDBY_bm | AC_LPMODE_bm)) | (enable ? AC_RUNSTDBY_bm : 0) | pwmode;
#else
      _ac->CTRLA = (_ac->CTRLA & ~AC_RUNSTDBY_bm) | (enable ? AC_RUNSTDBY_bm : 0);
#endif
    }

    void output(bool enable = true)
    {
      _ac->CTRLA = (_ac->CTRLA & ~AC_OUTEN_bm) | (enable ? AC_OUTEN_bm : 0);
    }

#if defined(AC_WINSEL_gm)
    void winsel(WINSEL winsel = WINSEL_DISABLED)
    {
      _ac->CTRLB = winsel;
    }
#endif

    void vref(Vref::ACREF refsel, bool alwayson = false)
    {
      Vref::ac(_ch, refsel, alwayson);
    }

#if defined(AC_DACREF_gm)
    void dacref(uint8_t val = 0xFF)
    {
      _ac->DACREF = val;
    }
#endif

    void callback(callback_t func)
    {
      _callback = func;
    }

    void interrupt(INTMODE intmode, bool enable = true)
    {
      if (enable)
      {
        _ac->STATUS  = AC_CMPIF_bm;
#if defined(CONFIG_AC_ISR)
  #if defined(AC_POWER_gm)
        _ac->INTCTRL = (_ac->INTCTRL & ~AC_INTMODE_NORMAL_gm) | intmode | AC_CMP_bm;
  #else
        _ac->CTRLA   = (_ac->CTRLA & ~AC_INTMODE_gm) | intmode;
        _ac->INTCTRL = AC_CMP_bm;
  #endif
#endif
      }
      else
        _ac->INTCTRL &= ~AC_CMP_bm;
    }

    STATUS status(void)
    {
      return (STATUS)_ac->STATUS;
    }

    static void poll(void)
    {
#if !CONFIG_AC_ISR
      for (uint8_t i = 0; i < AC_MAX_CH; ++i)
      {
        Ac* ac = _instances[i];
        if (ac)
          ac->isr_cmp();
      }
#endif
    }

  protected:

    static Ac *_instances[AC_MAX_CH];

    AC_t*      _ac;
    uint8_t    _ch;
    callback_t _callback;

#if CONFIG_AC_ISR
    friend void ac_isr_cmp(uint8_t ch);
#endif

    inline void isr_cmp(void)
    {
      uint8_t flags = _ac->STATUS = _ac->STATUS;
      if (flags & AC_CMPIF_bm)
      {
        if (_callback)
          _callback((STATUS)flags);
      }
    }
};
