/*
  avr8_adc.h - ADC Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_adc_dfp.h"
#include "avr8_clock.h"
#include "avr8_vref.h"
#include "avr8_fuse.h"

#if defined(ADC_MUXPOS_VDD10_gc)
  #define ADC_MUXPOS_VDDDIV10_gc ADC_MUXPOS_VDD10_gc
#endif
#if defined(ADC_SAMPNUM_ACC1_gc)
  #define ADC_SAMPNUM_NONE_gc ADC_SAMPNUM_ACC1_gc
#endif
#if defined(ADC_WINSRC_bm)
  #if !defined(ADC_WINSRC_RESULT_gc)
    #define ADC_WINSRC_RESULT_gc 0
  #endif
  #if !defined(ADC_WINSRC_SAMPLE_gc)
    #define ADC_WINSRC_SAMPLE_gc ADC_WINSRC_bm
  #endif
#endif

#if defined(ADC_RESSEL_bm)
  #define ADC_RESSEL_gm ADC_RESSEL_bm
#endif

class Adc
{
  public:

    /* Clock Pre-scaler select */
    typedef enum
    {
#if defined(__AVR_0__) || defined(__AVR_1__)
      PRESC_DIV2   = ADC_PRESC_DIV2_gc,    /* CLK_PER divided by 2 */
      PRESC_DIV4   = ADC_PRESC_DIV4_gc,    /* CLK_PER divided by 4 */
      PRESC_DIV8   = ADC_PRESC_DIV8_gc,    /* CLK_PER divided by 8 */
      PRESC_DIV16  = ADC_PRESC_DIV16_gc,   /* CLK_PER divided by 16 */
      PRESC_DIV32  = ADC_PRESC_DIV32_gc,   /* CLK_PER divided by 32 */
      PRESC_DIV64  = ADC_PRESC_DIV64_gc,   /* CLK_PER divided by 64 */
      PRESC_DIV128 = ADC_PRESC_DIV128_gc,  /* CLK_PER divided by 128 */
      PRESC_DIV256 = ADC_PRESC_DIV256_gc,  /* CLK_PER divided by 256 */
#elif defined(__AVR_DA__) || defined(__AVR_DB__) || defined(__AVR_DD__)
      PRESC_DIV2   = ADC_PRESC_DIV2_gc,    /* CLK_PER divided by 2 */
      PRESC_DIV4   = ADC_PRESC_DIV4_gc,    /* CLK_PER divided by 4 */
      PRESC_DIV8   = ADC_PRESC_DIV8_gc,    /* CLK_PER divided by 8 */
      PRESC_DIV12  = ADC_PRESC_DIV12_gc,   /* CLK_PER divided by 12 */
      PRESC_DIV16  = ADC_PRESC_DIV16_gc,   /* CLK_PER divided by 16 */
      PRESC_DIV20  = ADC_PRESC_DIV20_gc,   /* CLK_PER divided by 20 */
      PRESC_DIV24  = ADC_PRESC_DIV24_gc,   /* CLK_PER divided by 24 */
      PRESC_DIV28  = ADC_PRESC_DIV28_gc,   /* CLK_PER divided by 28 */
      PRESC_DIV32  = ADC_PRESC_DIV32_gc,   /* CLK_PER divided by 32 */
      PRESC_DIV48  = ADC_PRESC_DIV48_gc,   /* CLK_PER divided by 48 */
      PRESC_DIV64  = ADC_PRESC_DIV64_gc,   /* CLK_PER divided by 64 */
      PRESC_DIV96  = ADC_PRESC_DIV96_gc,   /* CLK_PER divided by 96 */
      PRESC_DIV128 = ADC_PRESC_DIV128_gc,  /* CLK_PER divided by 128 */
      PRESC_DIV256 = ADC_PRESC_DIV256_gc,  /* CLK_PER divided by 256 */
#elif defined(__AVR_2__) || defined(__AVR_DU__) || defined(__AVR_EA__) || defined(__AVR_EB__)
      PRESC_DIV2   = ADC_PRESC_DIV2_gc,    /* CLK_PER divided by 2 */
      PRESC_DIV4   = ADC_PRESC_DIV4_gc,    /* CLK_PER divided by 4 */
      PRESC_DIV6   = ADC_PRESC_DIV6_gc,    /* CLK_PER divided by 6 */
      PRESC_DIV8   = ADC_PRESC_DIV8_gc,    /* CLK_PER divided by 8 */
      PRESC_DIV10  = ADC_PRESC_DIV10_gc,   /* CLK_PER divided by 10 */
      PRESC_DIV12  = ADC_PRESC_DIV12_gc,   /* CLK_PER divided by 12 */
      PRESC_DIV14  = ADC_PRESC_DIV14_gc,   /* CLK_PER divided by 14 */
      PRESC_DIV16  = ADC_PRESC_DIV16_gc,   /* CLK_PER divided by 16 */
      PRESC_DIV20  = ADC_PRESC_DIV20_gc,   /* CLK_PER divided by 20 */
      PRESC_DIV24  = ADC_PRESC_DIV24_gc,   /* CLK_PER divided by 24 */
      PRESC_DIV28  = ADC_PRESC_DIV28_gc,   /* CLK_PER divided by 28 */
      PRESC_DIV32  = ADC_PRESC_DIV32_gc,   /* CLK_PER divided by 32 */
      PRESC_DIV40  = ADC_PRESC_DIV40_gc,   /* CLK_PER divided by 40 */
      PRESC_DIV48  = ADC_PRESC_DIV48_gc,   /* CLK_PER divided by 48 */
      PRESC_DIV56  = ADC_PRESC_DIV56_gc,   /* CLK_PER divided by 56 */
      PRESC_DIV64  = ADC_PRESC_DIV64_gc,   /* CLK_PER divided by 64 */
#endif
    } PRESC;

    /* Analog Channel Selection Bits select */
    typedef enum
    {
#if defined(ADC_MUXPOS_AIN0_gc)
      MUXPOS_AIN0      = ADC_MUXPOS_AIN0_gc,       /* ADC input pin 0 */
#endif
#if defined(ADC_MUXPOS_AIN1_gc)
      MUXPOS_AIN1      = ADC_MUXPOS_AIN1_gc,       /* ADC input pin 1 */
      MUXPOS_AIN2      = ADC_MUXPOS_AIN2_gc,       /* ADC input pin 2 */
      MUXPOS_AIN3      = ADC_MUXPOS_AIN3_gc,       /* ADC input pin 3 */
#endif
      MUXPOS_AIN4      = ADC_MUXPOS_AIN4_gc,       /* ADC input pin 4 */
      MUXPOS_AIN5      = ADC_MUXPOS_AIN5_gc,       /* ADC input pin 5 */
      MUXPOS_AIN6      = ADC_MUXPOS_AIN6_gc,       /* ADC input pin 6 */
      MUXPOS_AIN7      = ADC_MUXPOS_AIN7_gc,       /* ADC input pin 7 */
#if defined(ADC_MUXPOS_AIN8_gc)
      MUXPOS_AIN8      = ADC_MUXPOS_AIN8_gc,       /* ADC input pin 8 */
      MUXPOS_AIN9      = ADC_MUXPOS_AIN9_gc,       /* ADC input pin 9 */
#endif
#if defined(ADC_MUXPOS_AIN10_gc)
      MUXPOS_AIN10     = ADC_MUXPOS_AIN10_gc,      /* ADC input pin 10 */
      MUXPOS_AIN11     = ADC_MUXPOS_AIN11_gc,      /* ADC input pin 11 */
#endif
#if defined(ADC_MUXPOS_AIN12_gc)
      MUXPOS_AIN12     = ADC_MUXPOS_AIN12_gc,      /* ADC input pin 12 */
      MUXPOS_AIN13     = ADC_MUXPOS_AIN13_gc,      /* ADC input pin 13 */
      MUXPOS_AIN14     = ADC_MUXPOS_AIN14_gc,      /* ADC input pin 14 */
      MUXPOS_AIN15     = ADC_MUXPOS_AIN15_gc,      /* ADC input pin 15 */
#endif
#if defined(ADC_MUXPOS_AIN16_gc)
      MUXPOS_AIN16     = ADC_MUXPOS_AIN16_gc,      /* ADC input pin 16 */
      MUXPOS_AIN17     = ADC_MUXPOS_AIN17_gc,      /* ADC input pin 17 */
#endif
#if defined(ADC_MUXPOS_AIN18_gc)
      MUXPOS_AIN18     = ADC_MUXPOS_AIN18_gc,      /* ADC input pin 18 */
      MUXPOS_AIN19     = ADC_MUXPOS_AIN19_gc,      /* ADC input pin 19 */
      MUXPOS_AIN20     = ADC_MUXPOS_AIN20_gc,      /* ADC input pin 29 */
      MUXPOS_AIN21     = ADC_MUXPOS_AIN21_gc,      /* ADC input pin 21 */
#endif
#if defined(ADC_MUXPOS_AIN22_gc)
      MUXPOS_AIN22     = ADC_MUXPOS_AIN22_gc,      /* ADC input pin 22 */
      MUXPOS_AIN23     = ADC_MUXPOS_AIN23_gc,      /* ADC input pin 23 */
      MUXPOS_AIN24     = ADC_MUXPOS_AIN24_gc,      /* ADC input pin 24 */
      MUXPOS_AIN25     = ADC_MUXPOS_AIN25_gc,      /* ADC input pin 25 */
      MUXPOS_AIN26     = ADC_MUXPOS_AIN26_gc,      /* ADC input pin 26 */
      MUXPOS_AIN27     = ADC_MUXPOS_AIN27_gc,      /* ADC input pin 27 */
#endif
#if defined(ADC_MUXPOS_AIN28_gc)
      MUXPOS_AIN28     = ADC_MUXPOS_AIN28_gc,      /* ADC input pin 28 */
      MUXPOS_AIN29     = ADC_MUXPOS_AIN29_gc,      /* ADC input pin 29 */
      MUXPOS_AIN30     = ADC_MUXPOS_AIN30_gc,      /* ADC input pin 30 */
      MUXPOS_AIN31     = ADC_MUXPOS_AIN31_gc,      /* ADC input pin 31 */
#endif
      MUXPOS_GND         = ADC_MUXPOS_GND_gc,         /* GND */
      MUXPOS_TEMPSENSE   = ADC_MUXPOS_TEMPSENSE_gc,   /* Temp sensor */
#if defined(ADC_MUXPOS_INTREF_gc)
      MUXPOS_INTREF      = ADC_MUXPOS_INTREF_gc,      /* Internal Ref */
#endif
#if defined(ADC_MUXPOS_VDDDIV10_gc)
      MUXPOS_VDDDIV10    = ADC_MUXPOS_VDDDIV10_gc,    /* VDD/10 */
#endif
#if defined(ADC_MUXPOS_VDDIO2DIV10_gc)
      MUXPOS_VDDIO2DIV10 = ADC_MUXPOS_VDDIO2DIV10_gc, /* VDDIO2/10 */
#endif
#if defined(ADC_MUXPOS_DAC0_gc)
      MUXPOS_DAC0        = ADC_MUXPOS_DAC0_gc,        /* DAC0 Output */
#endif
#if defined(ADC_MUXPOS_DACREF0_gc)
      MUXPOS_DACREF0     = ADC_MUXPOS_DACREF0_gc,     /* DACREF from AC0 */
#endif
#if defined(ADC_MUXPOS_DACREF1_gc)
      MUXPOS_DACREF1     = ADC_MUXPOS_DACREF1_gc,     /* DACREF from AC1 */
      MUXPOS_DACREF2     = ADC_MUXPOS_DACREF2_gc,     /* DACREF from AC2 */
#endif
    } MUXPOS;

#if defined(ADC_MUXNEG_gm)
    /* Analog Channel Selection Bits select */
    typedef enum
    {
  #if defined(ADC_MUXNEG_AIN0_gc)
      MUXNEG_AIN0      = ADC_MUXNEG_AIN0_gc,       /* ADC input pin 0 */
  #endif
  #if defined(ADC_MUXNEG_AIN1_gc)
      MUXNEG_AIN1      = ADC_MUXNEG_AIN1_gc,       /* ADC input pin 1 */
      MUXNEG_AIN2      = ADC_MUXNEG_AIN2_gc,       /* ADC input pin 2 */
      MUXNEG_AIN3      = ADC_MUXNEG_AIN3_gc,       /* ADC input pin 3 */
  #endif
      MUXNEG_AIN4      = ADC_MUXNEG_AIN4_gc,       /* ADC input pin 4 */
      MUXNEG_AIN5      = ADC_MUXNEG_AIN5_gc,       /* ADC input pin 5 */
      MUXNEG_AIN6      = ADC_MUXNEG_AIN6_gc,       /* ADC input pin 6 */
      MUXNEG_AIN7      = ADC_MUXNEG_AIN7_gc,       /* ADC input pin 7 */
  #if defined(ADC_MUXNEG_AIN8_gc)
      MUXNEG_AIN8      = ADC_MUXNEG_AIN8_gc,       /* ADC input pin 8 */
      MUXNEG_AIN9      = ADC_MUXNEG_AIN9_gc,       /* ADC input pin 9 */
      MUXNEG_AIN10     = ADC_MUXNEG_AIN10_gc,      /* ADC input pin 10 */
      MUXNEG_AIN11     = ADC_MUXNEG_AIN11_gc,      /* ADC input pin 11 */
  #endif
  #if defined(ADC_MUXNEG_AIN12_gc)
      MUXNEG_AIN12     = ADC_MUXNEG_AIN12_gc,      /* ADC input pin 12 */
      MUXNEG_AIN13     = ADC_MUXNEG_AIN13_gc,      /* ADC input pin 13 */
      MUXNEG_AIN14     = ADC_MUXNEG_AIN14_gc,      /* ADC input pin 14 */
      MUXNEG_AIN15     = ADC_MUXNEG_AIN15_gc,      /* ADC input pin 15 */
  #endif
  #if defined(ADC_MUXNEG_AIN16_gc)
      MUXNEG_AIN16     = ADC_MUXNEG_AIN16_gc,      /* ADC input pin 16 */
      MUXNEG_AIN17     = ADC_MUXNEG_AIN17_gc,      /* ADC input pin 17 */
  #endif
  #if defined(ADC_MUXNEG_AIN18_gc)
      MUXNEG_AIN18     = ADC_MUXNEG_AIN18_gc,      /* ADC input pin 18 */
      MUXNEG_AIN19     = ADC_MUXNEG_AIN19_gc,      /* ADC input pin 19 */
      MUXNEG_AIN20     = ADC_MUXNEG_AIN20_gc,      /* ADC input pin 20 */
      MUXNEG_AIN21     = ADC_MUXNEG_AIN21_gc,      /* ADC input pin 21 */
  #endif
  #if defined(ADC_MUXNEG_AIN22_gc)
      MUXNEG_AIN22     = ADC_MUXNEG_AIN22_gc,      /* ADC input pin 22 */
      MUXNEG_AIN23     = ADC_MUXNEG_AIN23_gc,      /* ADC input pin 23 */
      MUXNEG_AIN24     = ADC_MUXNEG_AIN24_gc,      /* ADC input pin 24 */
      MUXNEG_AIN25     = ADC_MUXNEG_AIN25_gc,      /* ADC input pin 25 */
      MUXNEG_AIN26     = ADC_MUXNEG_AIN26_gc,      /* ADC input pin 26 */
      MUXNEG_AIN27     = ADC_MUXNEG_AIN27_gc,      /* ADC input pin 27 */
  #endif
  #if defined(ADC_MUXNEG_AIN28_gc)
      MUXNEG_AIN28     = ADC_MUXNEG_AIN28_gc,      /* ADC input pin 28 */
      MUXNEG_AIN29     = ADC_MUXNEG_AIN29_gc,      /* ADC input pin 29 */
      MUXNEG_AIN30     = ADC_MUXNEG_AIN30_gc,      /* ADC input pin 30 */
      MUXNEG_AIN31     = ADC_MUXNEG_AIN31_gc,      /* ADC input pin 31 */
  #endif
      MUXNEG_GND       = ADC_MUXNEG_GND_gc,        /* GND */
  #if defined(ADC_MUXNEG_VDDDIV10_gc)
      MUXNEG_VDDDIV10  = ADC_MUXNEG_VDDDIV10_gc,   /* VDD/10 */
  #endif
  #if defined(ADC_MUXNEG_DACREF0_gc)
      MUXNEG_DACREF0   = ADC_MUXNEG_DACREF0_gc,    /* DACREF from AC0 */
  #endif
  #if defined(ADC_MUXNEG_DACREF1_gc)
      MUXNEG_DACREF1   = ADC_MUXNEG_DACREF1_gc,    /* DACREF from AC1 */
  #endif
    } MUXNEG;
#endif

    /* Sample numbers select */
    typedef enum ADC_SAMPNUM_enum
    {
      SAMPNUM_NONE    = ADC_SAMPNUM_NONE_gc,     /* No accumulation */
      SAMPNUM_ACC2    = ADC_SAMPNUM_ACC2_gc,     /* 2 samples accumulation */
      SAMPNUM_ACC4    = ADC_SAMPNUM_ACC4_gc,     /* 4 samples accumulation */
      SAMPNUM_ACC8    = ADC_SAMPNUM_ACC8_gc,     /* 8 samples accumulation */
      SAMPNUM_ACC16   = ADC_SAMPNUM_ACC16_gc,    /* 16 samples accumulation */
      SAMPNUM_ACC32   = ADC_SAMPNUM_ACC32_gc,    /* 32 samples accumulation */
      SAMPNUM_ACC64   = ADC_SAMPNUM_ACC64_gc,    /* 64 samples accumulation */
#if defined(ADC_SAMPNUM_ACC128_gc)
      SAMPNUM_ACC128  = ADC_SAMPNUM_ACC128_gc,   /* 128 samples accumulation */
#endif
#if defined(ADC_SAMPNUM_ACC256_gc)
      SAMPNUM_ACC256  = ADC_SAMPNUM_ACC256_gc,   /* 256 samples accumulation */
      SAMPNUM_ACC512  = ADC_SAMPNUM_ACC512_gc,   /* 512 samples accumulation */
      SAMPNUM_ACC1024 = ADC_SAMPNUM_ACC1024_gc,  /* 1024 samples accumulation */
#endif
    } SAMPNUM;

#if defined(ADC_SAMPDLY_gm)
    /* Initial Delay Selection select */
    typedef enum
    {
      INITDLY_DLY0   = ADC_INITDLY_DLY0_gc,    /* Delay 0 CLK_ADC cycles */
      INITDLY_DLY16  = ADC_INITDLY_DLY16_gc,   /* Delay 16 CLK_ADC cycles */
      INITDLY_DLY32  = ADC_INITDLY_DLY32_gc,   /* Delay 32 CLK_ADC cycles */
      INITDLY_DLY64  = ADC_INITDLY_DLY64_gc,   /* Delay 64 CLK_ADC cycles */
      INITDLY_DLY128 = ADC_INITDLY_DLY128_gc,  /* Delay 128 CLK_ADC cycles */
      INITDLY_DLY256 = ADC_INITDLY_DLY256_gc,  /* Delay 256 CLK_ADC cycles */
    } INITDLY;

    /* Automatic Sampling Delay Variation select */
    typedef enum
    {
  #if defined(ADC_ASDV_bm)
      ASDV_ASVOFF = ADC_ASDV_ASVOFF_gc,  /* The Automatic Sampling Delay Variation is disabled */
      ASDV_ASVON  = ADC_ASDV_ASVON_gc,   /* The Automatic Sampling Delay Variation is enabled */
  #else
      ASDV_ASVOFF = 0,
  #endif
    } ASDV;
#endif

#if defined(ADC_GAIN_gm)
    /* Gain select */
    typedef enum
    {
      GAIN_1X  = ADC_GAIN_1X_gc,   /* 1X Gain */
      GAIN_2X  = ADC_GAIN_2X_gc,   /* 2X Gain */
      GAIN_4X  = ADC_GAIN_4X_gc,   /* 4X Gain */
      GAIN_8X  = ADC_GAIN_8X_gc,   /* 8X Gain */
      GAIN_16X = ADC_GAIN_16X_gc,  /* 16X Gain */
    } GAIN;
#endif

#if defined(ADC_PGABIASSEL_gm)
    /* PGA BIAS Select */
    typedef enum
    {
      PGABIASSEL_100PCT = ADC_PGABIASSEL_100PCT_gc,  /* 100% BIAS current. */
      PGABIASSEL_75PCT  = ADC_PGABIASSEL_75PCT_gc,   /* 75% BIAS current. Usable for CLK_ADC<4.5MHz */
      PGABIASSEL_50PCT  = ADC_PGABIASSEL_50PCT_gc,   /* 50% BIAS current. Usable for CLK_ADC<3MHz */
      PGABIASSEL_25PCT  = ADC_PGABIASSEL_25PCT_gc,   /* 25% BIAS current. Usable for CLK_ADC<1.5MHz */
    } PGABIASSEL;
#endif

#if defined(ADC_ADCPGASAMPDUR_gm)
    /* ADC PGA Sample Duration select */
    typedef enum
    {
      ADCPGASAMPDUR_6CLK    = ADC_ADCPGASAMPDUR_6CLK_gc,   /* 6 ADC cycles */
      ADCPGASAMPDUR_15CLK   = ADC_ADCPGASAMPDUR_15CLK_gc,  /* 15 ADC cycles */
      ADCPGASAMPDUR_20CLK   = ADC_ADCPGASAMPDUR_20CLK_gc,  /* 20 ADC cycles (default) */
    } ADCPGASAMPDUR;
#endif

    /* Window Comparator Mode select */
    typedef enum
    {
      WINCM_NONE    = ADC_WINCM_NONE_gc,     /* No Window Comparison */
      WINCM_BELOW   = ADC_WINCM_BELOW_gc,    /* Below Window */
      WINCM_ABOVE   = ADC_WINCM_ABOVE_gc,    /* Above Window */
      WINCM_INSIDE  = ADC_WINCM_INSIDE_gc,   /* Inside Window */
      WINCM_OUTSIDE = ADC_WINCM_OUTSIDE_gc,  /* Outside Window */
    } WINCM;

    /* Window Mode Source select */
    typedef enum
    {
#if defined(ADC_WINSRC_bm)
      WINSRC_RESULT = ADC_WINSRC_RESULT_gc,  /* Result register used as Window Comparator Source */
      WINSRC_SAMPLE = ADC_WINSRC_SAMPLE_gc,  /* Sample register used as Window Comparator Source */
#else
      WINSRC_RESULT = 0,
#endif
    } WINSRC;

    /* ADC.INTFLAGS  bit masks and bit positions */
    typedef enum
    {
      INTFLAGS_RESRDY  = ADC_RESRDY_bm,    /* Result Ready Interrupt Enable bit mask. */
#if defined(ADC_SAMPRDY_bm)
      INTFLAGS_SAMPRDY = ADC_SAMPRDY_bm,   /* Sample Ready Interrupt Enable bit mask. */
#endif
      INTFLAGS_WCMP    = ADC_WCMP_bm,      /* Window Comparator Interrupt Enable bit mask. */
#if defined(ADC_RESOVR_bm)
      INTFLAGS_RESOVR  = ADC_RESOVR_bm,    /* Result Overwritten Interrupt Enable bit mask. */
#endif
#if defined(ADC_SAMPOVR_bm)
      INTFLAGS_SAMPOVR = ADC_SAMPOVR_bm,   /* Sample Overwritten Interrupt Enable bit mask. */
#endif
#if defined(ADC_TRIGOVR_bm)
      INTFLAGS_TRIGOVR = ADC_TRIGOVR_bm,   /* Trigger Overrun Interrupt Enable bit mask. */
#endif
    } INTFLAGS;

    /* Mode select */
    typedef enum
    {
#if defined(ADC_MODE_gm)
      MODE_SINGLE_8BIT    = ADC_MODE_SINGLE_8BIT_gc,     /* Single Conversion 8-bit */
  #if defined(ADC_MODE_SINGLE_10BIT_gc)
      MODE_SINGLE_10BIT   = ADC_MODE_SINGLE_10BIT_gc,    /* Single Conversion 10-bit */
  #endif
  #if defined(ADC_MODE_SINGLE_12BIT_gc)
      MODE_SINGLE_12BIT   = ADC_MODE_SINGLE_12BIT_gc,    /* Single Conversion 12-bit */
  #endif
      MODE_SERIES         = ADC_MODE_SERIES_gc,          /* Series Accumulation */
  #if defined(ADC_MODE_SERIES_SCALING_gc)
      MODE_SERIES_SCALING = ADC_MODE_SERIES_SCALING_gc,  /* Series Accumulation with Scaling */
  #endif
      MODE_BURST          = ADC_MODE_BURST_gc,           /* Burst Accumulation */
  #if defined(ADC_MODE_BURST_SCALING_gc)
      MODE_BURST_SCALING  = ADC_MODE_BURST_SCALING_gc,   /* Burst Accumulation with Scaling */
  #endif
  #if defined(ADC_MODE_ACCTEST_gc)
      MODE_ACCTEST        = ADC_MODE_ACCTEST_gc,         /* Acc test mode for FuSa */
  #endif
  #if defined(ADC_MODE_SINGLE_12BIT_gc)
      MODE_DEFAULT        = ADC_MODE_SINGLE_12BIT_gc,    /* Single Conversion 12-bit */
  #elif defined(ADC_MODE_SINGLE_10BIT_gc)
      MODE_DEFAULT        = ADC_MODE_SINGLE_10BIT_gc,    /* Single Conversion 10-bit */
  #else
      MODE_DEFAULT        = ADC_MODE_SINGLE_8BIT_gc,     /* Single Conversion 8-bit */
  #endif
#else
  #if defined(ADC_RESSEL_8BIT_gc)
      MODE_SINGLE_8BIT    = ADC_RESSEL_8BIT_gc,          /* 8-bit mode */
  #endif
  #if defined(ADC_RESSEL_10BIT_gc)
      MODE_SINGLE_10BIT   = ADC_RESSEL_10BIT_gc,         /* 10-bit mode */
  #endif
  #if defined(ADC_RESSEL_12BIT_gc)
      MODE_SINGLE_12BIT   = ADC_RESSEL_12BIT_gc,         /* 12-bit mode */
  #endif
  #if defined(ADC_RESSEL_12BIT_gc)
      MODE_DEFAULT        = ADC_RESSEL_12BIT_gc,         /* 12-bit mode */
  #elif defined(ADC_RESSEL_10BIT_gc)
      MODE_DEFAULT        = ADC_RESSEL_10BIT_gc,         /* 10-bit mode */
  #else
      MODE_DEFAULT        = ADC_RESSEL_8BIT_gc,          /* 8-bit mode */
  #endif
#endif
    } MODE;

    /* Start command select */
    typedef enum
    {
#if defined(ADC_STCONV_bm)
      START_STOP          = 0,                           /* Stop an ongoing conversion */
      START_IMMEDIATE     = ADC_STCONV_bm,               /* Start immediately */
#else
      START_STOP          = ADC_START_STOP_gc,           /* Stop an ongoing conversion */
      START_IMMEDIATE     = ADC_START_IMMEDIATE_gc,      /* Start immediately */
  #if defined(ADC_START_MUXPOS_WRITE_gc)
      START_MUXPOS_WRITE  = ADC_START_MUXPOS_WRITE_gc,   /* Start on MUXPOS write */
  #endif
  #if defined(ADC_START_MUXNEG_WRITE_gc)
      START_MUXNEG_WRITE  = ADC_START_MUXNEG_WRITE_gc,   /* Start on MUXNEG write */
  #endif
      START_EVENT_TRIGGER = ADC_START_EVENT_TRIGGER_gc,  /* Start on event */
#endif
    } START;

    Adc(ADC_t& adc) : _adc(&adc), _ch((&adc - &ADC0) / sizeof(adc)), _dlyclk(0), _status(0), _result(0)
    {
    }

    /* virtual */ ~Adc(void)
    {
      end();
    }

    void begin(Vref::ADCREF refsel, uint32_t frq = CONFIG_ADC_CLOCK)
    {
      begin(refsel, hz2ps(frq));
    }

    void begin(Vref::ADCREF refsel, PRESC presc)
    {
      end();
      _status = 0;
      _result = 0;
#if defined(ADC_DUTYCYC_bm)
      _adc->CALIB    = 0;
#endif
      _adc->WINHT    = 0;
      _adc->WINLT    = 0;
      _adc->DBGCTRL  = 0;
      _adc->INTFLAGS = 0xFF;
      _adc->INTCTRL  = 0;
#if defined(ADC_STARTEI_bm)
      _adc->EVCTRL   = 0;
#endif
#if defined(ADC_MUXNEG_gm)
      _adc->MUXNEG   = 0;
#endif
      _adc->MUXPOS   = 0;
#if defined(ADC_PGAEN_bm)
  #if defined(ADC_ADCPGASAMPDUR_gm)
      _adc->PGACTRL  = ADC_ADCPGASAMPDUR_20CLK_gc;
  #else
      _adc->PGACTRL  = 0;
  #endif
#endif
      _adc->COMMAND  = 0;
#if defined(ADC_SAMPDUR_gm)
      _adc->CTRLF    = 0;
#endif
      _adc->CTRLE    = 0;
      _adc->CTRLD    = 0;
      _adc->CTRLC    = 0;
      _adc->CTRLB    = 0;
      prescaler(presc);
#if defined(ADC_SAMPDLY_gm)
      sampdly();
#endif
      Vref::adc(_ch, refsel);
      _adc->CTRLA    = ADC_ENABLE_bm;
    }

    void end(void)
    {
      _adc->CTRLA = 0;
    }

    void standby(bool enable = true)
    {
      _adc->CTRLA = (_adc->CTRLA & ~ADC_RUNSTDBY_bm) | (enable ? ADC_RUNSTDBY_bm : 0);
    }

    void freerun(bool enable = true)
    {
#if defined(ADC_SAMPDUR_gm)
      _adc->CTRLF = (_adc->CTRLF & ~ADC_FREERUN_bm) | (enable ? ADC_FREERUN_bm : 0);
#else
      _adc->CTRLA = (_adc->CTRLA & ~ADC_FREERUN_bm) | (enable ? ADC_FREERUN_bm : 0);
#endif
    }

#if defined(ADC_LOWLAT_bm)
    void lowlat(bool enable = true)
    {
      _adc->CTRLA = (_adc->CTRLA & ~ADC_LOWLAT_bm) | (enable ? ADC_LOWLAT_bm : 0);
    }
#endif

#if defined(ADC_LEFTADJ_bm)
    void leftadj(bool enable = true)
    {
  #if defined(ADC_SAMPDUR_gm)
      _adc->CTRLF = (_adc->CTRLF & ~ADC_LEFTADJ_bm) | (enable? ADC_LEFTADJ_bm : 0);
  #else
      _adc->CTRLA = (_adc->CTRLA & ~ADC_LEFTADJ_bm) | (enable? ADC_LEFTADJ_bm : 0);
  #endif
    }
#endif

#if defined(ADC_CHOPPING_bm)
    void chopping(bool enable = true)
    {
      _adc->CTRLF = (_adc->CTRLF & ~ADC_CHOPPING_bm) | (enable ? ADC_CHOPPING_ENABLE_gc : ADC_CHOPPING_DISABLE_gc);
    }
#endif

#if defined(ADC_PGAEN_bm)
    void muxpos(MUXPOS muxpos = MUXPOS_GND, bool pga = false)
    {
      _adc->MUXPOS = (_adc->MUXPOS & ~(ADC_MUXPOS_gm | ADC_VIA_gm)) | (pga ? ADC_VIA_PGA_gc : ADC_VIA_DIRECT_gc) | muxpos;
    }

    void muxneg(MUXNEG muxneg = MUXNEG_GND, bool pga = false)
    {
  #if defined(ADC_CONVMODE_gm)
      _adc->CTRLA = (_adc->CTRLA & ~ADC_CONVMODE_gm) | (muxneg != MUXNEG_GND ? ADC_CONVMODE_DIFF_gc : ADC_CONVMODE_SINGLEENDED_gc);
  #else
      _adc->COMMAND = (_adc->COMMAND & ~ADC_DIFF_bm) | (muxneg != MUXNEG_GND ? ADC_DIFF_bm : 0);
  #endif
      _adc->MUXNEG = (_adc->MUXNEG & ~(ADC_MUXNEG_gm | ADC_VIA_gm)) | (pga ? ADC_VIA_PGA_gc : ADC_VIA_DIRECT_gc) | muxneg;
    }

  #if defined(ADC_ADCPGASAMPDUR_gm)
    void pga(bool enable = false, GAIN gain = GAIN_1X, PGABIASSEL bias = PGABIASSEL_100PCT, ADCPGASAMPDUR sampdur = ADCPGASAMPDUR_20CLK)
    {
      if (!enable)
      {
        _adc->MUXPOS &= ~ADC_VIA_gm;
        _adc->MUXNEG &= ~ADC_VIA_gm;
      }
      _adc->PGACTRL = gain | bias | sampdur | (enable ? ADC_PGAEN_bm : 0);
    }
  #else
    void pga(bool enable = false, GAIN gain = GAIN_1X, PGABIASSEL bias = PGABIASSEL_100PCT)
    {
      if (!enable)
      {
        _adc->MUXPOS &= ~ADC_VIA_gm;
        _adc->MUXNEG &= ~ADC_VIA_gm;
      }
      _adc->PGACTRL = gain | bias | (enable ? ADC_PGAEN_bm : 0);
    }
  #endif
#else
    void muxpos(MUXPOS muxpos = MUXPOS_GND)
    {
      _adc->MUXPOS = (_adc->MUXPOS & ~ADC_MUXPOS_gm) | muxpos;
    }

  #if defined(ADC_MUXNEG_gm)
    void muxneg(MUXNEG muxneg = MUXNEG_GND)
    {
    #if defined(ADC_CONVMODE_gm)
      _adc->CTRLA = (_adc->CTRLA & ~ADC_CONVMODE_gm) | (muxneg != MUXNEG_GND ? ADC_CONVMODE_DIFF_gc : ADC_CONVMODE_SINGLEENDED_gc);
    #else
      _adc->COMMAND = (_adc->COMMAND & ~ADC_DIFF_bm) | (muxneg != MUXNEG_GND ? ADC_DIFF_bm : 0);
    #endif
      _adc->MUXNEG = (_adc->MUXNEG & ~ADC_MUXNEG_gm) | muxneg;
    }
  #endif
#endif

    void sampnum(SAMPNUM value = SAMPNUM_NONE)
    {
#if defined(ADC_SAMPDUR_gm)
      _adc->CTRLF = (_adc->CTRLF & ~ADC_SAMPNUM_gm) | value;
#else
      _adc->CTRLB = value;
#endif
    }

#if defined(ADC_SAMPDLY_gm)
    void sampdly(INITDLY init = INITDLY_DLY0, uint8_t samp = 0, ASDV asdv = ASDV_ASVOFF)
    {
      if (init == INITDLY_DLY0)
      {
        uint8_t d = _dlyclk;
        uint8_t i = 0;
        d >>= 4;
        do
          ++i; 
        while (d >>= 1);
        init = (INITDLY)(i << ADC_INITDLY_gp);
      }
      _adc->CTRLD = init | asdv | (samp & ADC_SAMPDLY_gm);
    }
#endif

#if defined(ADC_SAMPLEN_gm)
    void samplen(uint8_t value = 0)
    {
      _adc->SAMPCTRL = value;
    }
#endif

#if defined(ADC_SAMPDUR_gm)
    void sampdur(uint8_t value = 0)
    {
      _adc->CTRLE = value;
    }
#endif

    void window(uint16_t winlt = 0, uint16_t winht = 0, WINCM wincm = WINCM_NONE, WINSRC src = WINSRC_RESULT)
    {
      _adc->WINLT = winlt;
      _adc->WINHT = winht;
#if defined(ADC_WINSRC_bm)
      _adc->CTRLD = wincm | src;
#else
      _adc->CTRLE = wincm;
#endif
    }

#if defined(ADC_STARTEI_bm)
    void evctrl(bool startei = false)
    {
      _adc->EVCTRL = startei ? ADC_STARTEI_bm : 0;
    }
#endif

    void dbgctrl(bool dbgrun = true)
    {
      _adc->DBGCTRL = dbgrun ? ADC_DBGRUN_bm : 0;
    }

    void convert(START start = START_IMMEDIATE, MODE mode = MODE_DEFAULT)
    {
#if defined(ADC_MODE_gm)
      _adc->COMMAND = (_adc->COMMAND & ~(ADC_START_gm | ADC_MODE_gm)) | start | mode;
#else
      _adc->CTRLA = (_adc->CTRLA & ~ADC_RESSEL_gm) | mode ;
      _adc->COMMAND = (_adc->COMMAND & ~ADC_START_gm) | start;
#endif
      _status = start ? 0x80 : 0;
    }

    INTFLAGS status(void)
    {
      return (INTFLAGS)(_status | _adc->INTFLAGS);
    }

    bool converted(void)
    {
      return (status() & INTFLAGS_RESRDY) != 0;
    }

    int16_t result(bool wait = true)
    {
      if (status() & 0x80)
      {
        while (wait && !converted())
          yield();
        if (converted())
        {
#if defined(ADC0_RES)
          _result = _adc->RES;
#else
          _result = _adc->RESULT;
#endif
          _status = 0;
        }
      }
      return _result;
    }

    int16_t voltage(int16_t result, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      uint16_t res = resolution();
#if defined(ADC_MUXNEG_gm)
  #if defined(ADC_CONVMODE_gm)
      if ((_adc->CTRLA & ADC_CONVMODE_gm) == ADC_CONVMODE_DIFF_gc)
        res >>= 1;
  #else
      if (_adc->COMMAND & ADC_DIFF_bm)
        res >>= 1;
  #endif
#endif
      return (uint32_t)Vref::adc(_ch, vdd, ref) * result / res;
    }

    int16_t temperature(uint32_t frq = CONFIG_ADC_CLOCK)
    {
      return temperature(hz2ps(frq));
    }

    int16_t temperature(PRESC presc)
    {
#if defined(CLKCTRL_OSCHFCTRLA)
  #if defined(__AVR_DU__)
      const uint16_t SCALING_FACTOR = 1024;
  #else
      const uint16_t SCALING_FACTOR = 4096;
  #endif
      begin(Vref::ADCREF_2V048, presc);
      muxpos(MUXPOS_TEMPSENSE);
  #if defined(ADC_SAMPDUR_gm)
     _adc->CTRLE = _dlyclk;
  #else
      _adc->SAMPCTRL = _dlyclk >> 1;
  #endif
      convert();
  #if defined(__AVR_EA__) || defined(__AVR_EB__)
      return ((int32_t)(result() + (int16_t)Sigrow::tempsense1()) * Sigrow::tempsense0() + (SCALING_FACTOR / 2)) / SCALING_FACTOR - 273;
  #else
      return ((int32_t)((int16_t)Sigrow::tempsense1() - result()) * Sigrow::tempsense0() + (SCALING_FACTOR / 2)) / SCALING_FACTOR - 273;
  #endif
#elif defined(ADC_REFSEL_INTREF_gc)
      const uint16_t SCALING_FACTOR = 256;
      begin(Vref::ADCREF_1V1, presc);
      muxpos(MUXPOS_TEMPSENSE);
      _adc->SAMPCTRL = _dlyclk >> 1;
      convert();
      return ((int32_t)(result() - (int8_t)Sigrow::tempsense1()) * Sigrow::tempsense0() + (SCALING_FACTOR / 2)) / SCALING_FACTOR - 273;
#else
      const uint16_t SCALING_FACTOR = 256;
      begin(Vref::ADCREF_1V024, presc);
      muxpos(MUXPOS_TEMPSENSE);
      _adc->CTRLE = _dlyclk;
      convert();
      return ((int32_t)(result() - (int8_t)Sigrow::tempsense1()) * Sigrow::tempsense0() + (SCALING_FACTOR / 2)) / SCALING_FACTOR - 273;
#endif
    }

    int16_t cpuvdd(uint32_t frq = CONFIG_ADC_CLOCK)
    {
      return cpuvdd(hz2ps(frq));
    }

    int16_t cpuvdd(PRESC presc)
    {
#if defined(ADC_REFSEL_INTREF_gc)
      const uint16_t vref = 1100;
      Vref::adc(_ch, Vref::ADCREF_1V1);
      begin(Vref::ADCREF_VDD, presc);
      muxpos(MUXPOS_INTREF);
#else
      const uint16_t vref = 1024;
  #if defined(ADC_MUXPOS_DACREF0_gc)
      Vref::ac(0, Vref::ACREF_1V024);
      AC0.DACREF = 0xFF;
      begin(Vref::ADCREF_VDD, presc);
      muxpos(MUXPOS_DACREF0);
  #elif defined(ADC_MUXPOS_DAC0_gc)
      Vref::dac(0, Vref::DACREF_1V024);
      DAC0.CTRLA = DAC_ENABLE_bm;
      DAC0.DATA  = DAC_DATA_gm;
      begin(Vref::ADCREF_VDD, presc);
      muxpos(MUXPOS_DAC0);
  #endif
#endif
      convert();
      return (uint32_t)vref * resolution() / result();
    }

  protected:

    ADC_t*   _adc;
    uint8_t  _ch;
    uint8_t  _dlyclk;
    uint8_t  _status;
    uint16_t _result;

    static PRESC hz2ps(uint32_t frq)
    {
      static const uint8_t CLKDIV[] = {
#if defined(__AVR_0__) || defined(__AVR_1__)
        2, 4, 8, 16, 32, 64, 128, 0
#elif defined(__AVR_DA__) || defined(__AVR_DB__) || defined(__AVR_DD__)
        2, 4, 8, 12, 16, 20, 24, 28, 32, 48, 64, 96, 128, 0
#elif defined(__AVR_2__) || defined(__AVR_DU__) || defined(__AVR_EA__) || defined(__AVR_EB__)
        2, 4, 6, 8, 10, 12, 14, 16, 20, 24, 28, 32, 40, 48, 56, 64
#endif
      };
      uint32_t div = Clock::frequency() / frq;
      for (uint8_t i = 0; i < sizeof(CLKDIV); ++i)
      {
        uint8_t ps = CLKDIV[i];
        if (div <= (ps ? ps : 256))
          return (PRESC)(ADC_PRESC_DIV2_gc + (i << ADC_PRESC_gp));
      }
      return (PRESC)(ADC_PRESC_DIV2_gc + ((sizeof(CLKDIV) - 1) << ADC_PRESC_gp));
    }

    void prescaler(PRESC presc)
    {
#if defined(ADC_SAMPDUR_gm)
      _adc->CTRLB = (_adc->CTRLB & ~ADC_PRESC_gm) | presc;
  #if defined(ADC_TIMEBASE_gm)
      _adc->CTRLC = (_adc->CTRLC & ~ADC_TIMEBASE_gm) | ((uint8_t)(Clock::frequency() / 1000000UL) << ADC_TIMEBASE_gp);
  #endif
#else
      _adc->CTRLC = (_adc->CTRLC & ~ADC_PRESC_gm) | presc;
  #if defined(ADC_SAMPDLY_gm)
      uint32_t clock = Clock::frequency();
    #if defined(ADC_DUTYCYC_bm)
      _adc->CALIB = (clock >> (presc + 1)) <= 1500000UL ? ADC_DUTYCYC_bm : 0;
    #endif
      /* calculate 32us delay clock counts */
      int8_t shift = 5 - 1 - presc;
      _dlyclk = (shift >= 0 ? clock << shift : clock >> -shift) / 1000000UL;
  #endif
#endif
    }

    uint16_t resolution(void)
    {
      uint16_t res = 0;
#if defined(ADC_MODE_gm)
      switch (_adc->COMMAND & ADC_MODE_gm)
      {
        default:
  #if defined(ADC_MODE_SINGLE_12BIT_gc)
        case ADC_MODE_SINGLE_12BIT_gc: res = 4096; break;
  #endif
  #if defined(ADC_MODE_SINGLE_10BIT_gc)
        case ADC_MODE_SINGLE_10BIT_gc: res = 1024; break;
  #endif
  #if defined(ADC_MODE_SINGLE_8BIT_gc)
        case ADC_MODE_SINGLE_8BIT_gc : res =  256; break;
  #endif
      }
#else
      switch (_adc->CTRLA & ADC_RESSEL_gm)
      {
        default:
  #if defined(ADC_RESSEL_12BIT_gc)
        case ADC_RESSEL_12BIT_gc: res = 4096; break;
  #endif
  #if defined(ADC_RESSEL_10BIT_gc)
        case ADC_RESSEL_10BIT_gc: res = 1024; break;
  #endif
  #if defined(ADC_RESSEL_8BIT_gc)
        case ADC_RESSEL_8BIT_gc : res =  256; break;
  #endif
      }
#endif
      return res;
    }
};
