/*
  avr8_alarm.cpp - Alarm Library for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#include <stddef.h>
#include "avr8_alarm.h"

#if CONFIG_TASK_USE || CONFIG_ALARM_USE

#define lengthof(a) (sizeof(a)/sizeof(a[0]))

AlarmClass AlarmTimer;

Alarm& Alarm::interval(int32_t interval, bool fromnow)
{
  if (fromnow)
    _alarm = Timer.read() + interval;
  _interval = interval;
  return *this;
}

int32_t Alarm::interval(void)
{
  return _interval;
}

Alarm& Alarm::handler(Alarm::handler_t handler)
{
  _handler = handler;
  return *this;
}

Alarm::handler_t Alarm::handler(void)
{
  return _handler;
}

Alarm& Alarm::param(uint8_t num, void *val)
{
  if (num < lengthof(_params))
    _params[num] = val;
  return *this;
}

void *Alarm::param(uint8_t num)
{
  return num < lengthof(_params) ? _params[num] : NULL;
}

bool Alarm::expire(void)
{
  if ((int32_t)(Timer.read() - _alarm) < 0)
    return false;
  _alarm += _interval;
  return true;
}

AlarmClass::AlarmClass(void) : _alarm_list(0), _poll_mode(0)
{
}

void AlarmClass::begin(void)
{
  _poll_mode = !setupAlarmTimer();
}

bool AlarmClass::start(Alarm& alarm)
{
  if ((alarm._interval < 1) || !alarm._handler)
    return false;
  sreg_t state = disableAndSaveInterrupts();
  if (add(alarm) && !_poll_mode)
    startAlarmTimer(_alarm_list->_alarm - Timer.read());
  restoreInterrupts(state);
  return true;
}

void AlarmClass::cancel(Alarm& alarm)
{
  sreg_t state = disableAndSaveInterrupts();
  for (Alarm **p = &_alarm_list; *p; p = &(*p)->_next)
  {
    if (*p == &alarm)
    {
      *p = (*p)->_next;
      break;
    }
  }
  restoreInterrupts(state);
}

void AlarmClass::cancel(uint8_t num, void *arg)
{
  if ((num < lengthof(Alarm::_params)) && arg)
  {
    sreg_t state = disableAndSaveInterrupts();
    for (Alarm **p = &_alarm_list; *p; p = &(*p)->_next)
    {
      if ((*p)->_params[num] == arg)
      {
        *p = (*p)->_next;
        break;
      }
    }
    restoreInterrupts(state);
  }
}

bool AlarmClass::add(Alarm& alarm)
{
  Alarm **p;
  for (p = &_alarm_list; *p && ((int32_t)((*p)->_alarm - alarm._alarm) <= 0); p = &(*p)->_next)
    continue;
  alarm._next = *p;
  *p = &alarm;
  return p == &_alarm_list;
}

bool AlarmClass::timeup(void)
{
  Alarm *p = NULL;
  while (_alarm_list)
  {
    int32_t t = _alarm_list->_alarm - Timer.read();
    if (p || (t > 0))
    {
      if (!_poll_mode)
        startAlarmTimer(t);
      return true;
    }
    _alarm_list = (p = _alarm_list)->_next;
    if (p->_handler(*p))
    {
      p->_alarm += p->_interval;
      if (!add(*p))
        p = NULL;
    }
    else
      p = NULL;
  }
  return false;
}

void AlarmClass::handle(void)
{
  if (_poll_mode)
  {
    Timer.poll();
    sreg_t state = disableAndSaveInterrupts();
    timeup();
    restoreInterrupts(state);
  }
}

#endif
