/*
  avr8_clock.h - CLKCTRL Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_clock_dfp.h"
#include "avr8_fuse.h"

#if defined(CLKCTRL_PDIV_2X_gc)
  #define CLKCTRL_PDIV_DIV2_gc CLKCTRL_PDIV_2X_gc
#endif
#if defined(CLKCTRL_PDIV_4X_gc)
  #define CLKCTRL_PDIV_DIV4_gc CLKCTRL_PDIV_4X_gc
#endif
#if defined(CLKCTRL_PDIV_8X_gc)
  #define CLKCTRL_PDIV_DIV8_gc CLKCTRL_PDIV_8X_gc
#endif
#if defined(CLKCTRL_PDIV_16X_gc)
  #define CLKCTRL_PDIV_DIV16_gc CLKCTRL_PDIV_16X_gc
#endif
#if defined(CLKCTRL_PDIV_32X_gc)
  #define CLKCTRL_PDIV_DIV32_gc CLKCTRL_PDIV_32X_gc
#endif
#if defined(CLKCTRL_PDIV_64X_gc)
  #define CLKCTRL_PDIV_DIV64_gc CLKCTRL_PDIV_64X_gc
#endif
#if defined(CLKCTRL_PDIV_6X_gc)
  #define CLKCTRL_PDIV_DIV6_gc CLKCTRL_PDIV_6X_gc
#endif
#if defined(CLKCTRL_PDIV_10X_gc)
  #define CLKCTRL_PDIV_DIV10_gc CLKCTRL_PDIV_10X_gc
#endif
#if defined(CLKCTRL_PDIV_12X_gc)
  #define CLKCTRL_PDIV_DIV12_gc CLKCTRL_PDIV_12X_gc
#endif
#if defined(CLKCTRL_PDIV_24X_gc)
  #define CLKCTRL_PDIV_DIV24_gc CLKCTRL_PDIV_24X_gc
#endif
#if defined(CLKCTRL_PDIV_48X_gc)
  #define CLKCTRL_PDIV_DIV48_gc CLKCTRL_PDIV_48X_gc
#endif

#if defined(CLKCTRL_MULFAC_DISABLE_gc)
  #define CLKCTRL_MULFAC_OFF_gc CLKCTRL_MULFAC_DISABLE_gc
#endif

#if defined(CLKCTRL_SOURCE_bm)
  #define CLKCTRL_SOURCE_gm CLKCTRL_SOURCE_bm
  #if !defined(CLKCTRL_SOURCE_OSCHF_gc)
    #define CLKCTRL_SOURCE_OSCHF_gc  0
  #endif
  #if !defined(CLKCTRL_SOURCE_EXTCLK_gc)
    #define CLKCTRL_SOURCE_EXTCLK_gc 1
  #endif
#endif

#if defined(CLKCTRL_PLLCTRLA) && !defined(CLKCTRL_CLKSEL_PLL_gc)
  #define CLKCTRL_CLKSEL_PLL_gc CLKCTRL_CLKSEL_gm
#endif

#if defined(CLKCTRL_SELHF_CRYSTAL_gc)
  #define CLKCTRL_SELHF_XTAL_gc CLKCTRL_SELHF_CRYSTAL_gc
#endif
#if defined(CLKCTRL_SELHF_EXTCLOCK_gc)
  #define CLKCTRL_SELHF_EXTCLK_gc CLKCTRL_SELHF_EXTCLOCK_gc
#endif

#if defined(CLKCTRL_CSUTHF_256CYC_gc)
  #define CLKCTRL_CSUTHF_256_gc CLKCTRL_CSUTHF_256CYC_gc
#endif
#if defined(CLKCTRL_CSUTHF_1KCYC_gc)
  #define CLKCTRL_CSUTHF_1K_gc CLKCTRL_CSUTHF_1KCYC_gc
#endif
#if defined (CLKCTRL_CSUTHF_4KCYC_gc)
  #define CLKCTRL_CSUTHF_4K_gc CLKCTRL_CSUTHF_4KCYC_gc
#endif

class Clock
{
  public:

    /* clock select select */
    typedef enum
    {
#if defined(CLKCTRL_OSCHFCTRLA)
      CLKSEL_OSCHF   = CLKCTRL_CLKSEL_OSCHF_gc,    /* Internal high-frequency oscillator */
      CLKSEL_OSC32K  = CLKCTRL_CLKSEL_OSC32K_gc,   /* Internal 32.768 kHz oscillator */
      CLKSEL_XOSC32K = CLKCTRL_CLKSEL_XOSC32K_gc,  /* 32.768 kHz crystal oscillator */
      CLKSEL_EXTCLK  = CLKCTRL_CLKSEL_EXTCLK_gc,   /* External clock */
  #if defined(CLKCTRL_CLKSEL_PLL_gc)
      CLKSEL_PLL     = CLKCTRL_CLKSEL_PLL_gc,      /* External clock */
  #endif
#else
      CLKSEL_OSC20M    = CLKCTRL_CLKSEL_OSC20M_gc,     /* 20MHz internal oscillator */
      CLKSEL_OSCULP32K = CLKCTRL_CLKSEL_OSCULP32K_gc,  /* 32KHz internal Ultra Low Power oscillator */
      CLKSEL_EXTCLK    = CLKCTRL_CLKSEL_EXTCLK_gc,     /* External clock */
#endif
    } CLKSEL;

#if defined(CLKCTRL_FRQSEL_gm)
    /* Frequency select (OSCHF) */
    typedef enum
    {
      FRQSEL_1M  = CLKCTRL_FRQSEL_1M_gc,   /* 1 MHz system clock */
      FRQSEL_2M  = CLKCTRL_FRQSEL_2M_gc,   /* 2 MHz system clock */
      FRQSEL_3M  = CLKCTRL_FRQSEL_3M_gc,   /* 3 MHz system clock */
      FRQSEL_4M  = CLKCTRL_FRQSEL_4M_gc,   /* 4 MHz system clock (default) */
      FRQSEL_8M  = CLKCTRL_FRQSEL_8M_gc,   /* 8 MHz system clock */
      FRQSEL_12M = CLKCTRL_FRQSEL_12M_gc,  /* 12 MHz system clock */
      FRQSEL_16M = CLKCTRL_FRQSEL_16M_gc,  /* 16 MHz system clock */
      FRQSEL_20M = CLKCTRL_FRQSEL_20M_gc,  /* 20 MHz system clock */
      FRQSEL_24M = CLKCTRL_FRQSEL_24M_gc,  /* 24 MHz system clock */
    } FRQSEL;
#endif

    /* Autotune select (OSCHF) */
    typedef enum
    {
#if defined(CLKCTRL_USBPLLSTATUS)
      AUTOTUNE_OFF = CLKCTRL_AUTOTUNE_OFF_gc,  /* Automatic tuning disabled */
      AUTOTUNE_32K = CLKCTRL_AUTOTUNE_32K_gc,  /* Automatic tuning against XOSC32K enabled */
      AUTOTUNE_SOF = CLKCTRL_AUTOTUNE_SOF_gc,  /* Automatic tuning against USB SOF enabled */
#else
      AUTOTUNE_OFF = 0,                        /* Automatic tuning disabled */
      AUTOTUNE_32K = 1,                        /* Automatic tuning against XOSC32K enabled */
#endif
    } AUTOTUNE;

    /* Prescaler division select */
    typedef enum
    {
      PDIV_DIV1   = 0,
      PDIV_DIV2   = CLKCTRL_PDIV_DIV2_gc  | CLKCTRL_PEN_bm,  /* Divide by 2 */
      PDIV_DIV4   = CLKCTRL_PDIV_DIV4_gc  | CLKCTRL_PEN_bm,  /* Divide by 4 */
      PDIV_DIV8   = CLKCTRL_PDIV_DIV8_gc  | CLKCTRL_PEN_bm,  /* Divide by 8 */
      PDIV_DIV16  = CLKCTRL_PDIV_DIV16_gc | CLKCTRL_PEN_bm,  /* Divide by 16 */
      PDIV_DIV32  = CLKCTRL_PDIV_DIV32_gc | CLKCTRL_PEN_bm,  /* Divide by 32 */
      PDIV_DIV64  = CLKCTRL_PDIV_DIV64_gc | CLKCTRL_PEN_bm,  /* Divide by 64 */
      PDIV_DIV6   = CLKCTRL_PDIV_DIV6_gc  | CLKCTRL_PEN_bm,  /* Divide by 6 */
      PDIV_DIV10  = CLKCTRL_PDIV_DIV10_gc | CLKCTRL_PEN_bm,  /* Divide by 10 */
      PDIV_DIV12  = CLKCTRL_PDIV_DIV12_gc | CLKCTRL_PEN_bm,  /* Divide by 12 */
      PDIV_DIV24  = CLKCTRL_PDIV_DIV24_gc | CLKCTRL_PEN_bm,  /* Divide by 24 */
      PDIV_DIV48  = CLKCTRL_PDIV_DIV48_gc | CLKCTRL_PEN_bm,  /* Divide by 48 */
#if defined(CLKCTRL_PBDIV_bm)
      PDIV_DIV128 = PDIV_DIV32 | CLKCTRL_PBDIV_bm,
      PDIV_DIV256 = PDIV_DIV64 | CLKCTRL_PBDIV_bm,
#endif
    } PDIV;

#if defined(CLKCTRL_CSUT_gm)
    /* Crystal startup time select */
    typedef enum
    {
      CSUT_1K  = CLKCTRL_CSUT_1K_gc,   /* 1k cycles */
      CSUT_16K = CLKCTRL_CSUT_16K_gc,  /* 16k cycles */
      CSUT_32K = CLKCTRL_CSUT_32K_gc,  /* 32k cycles */
      CSUT_64K = CLKCTRL_CSUT_64K_gc,  /* 64k cycles */
    } CSUT;
#endif

#if defined(CLKCTRL_CSUTHF_gm)
    /* Start-up Time Select */
    typedef enum
    {
      CSUTHF_256 = CLKCTRL_CSUTHF_256_gc,  /* 256 XOSCHF cycles */
      CSUTHF_1K  = CLKCTRL_CSUTHF_1K_gc,   /* 1K XOSCHF cycles */
      CSUTHF_4K  = CLKCTRL_CSUTHF_4K_gc,   /* 4K XOSCHF cycles */
    } CSUTHF;
#endif

#if defined(CLKCTRL_FRQRANGE_gm)
    /* Frequency Range select */
    typedef enum
    {
      FRQRANGE_8M  = CLKCTRL_FRQRANGE_8M_gc,   /* Max 8 MHz XTAL Frequency */
      FRQRANGE_16M = CLKCTRL_FRQRANGE_16M_gc,  /* Max 16 MHz XTAL Frequency */
      FRQRANGE_24M = CLKCTRL_FRQRANGE_24M_gc,  /* Max 24 MHz XTAL Frequency */
      FRQRANGE_32M = CLKCTRL_FRQRANGE_32M_gc,  /* Max 32 MHz XTAL Frequency */
    } FRQRANGE;
#endif

#if defined(CLKCTRL_SELHF_bm)
    /* External Source Select */
    typedef enum
    {
      SELHF_XTAL   = CLKCTRL_SELHF_XTAL_gc,    /* External Crystal */
      SELHF_EXTCLK = CLKCTRL_SELHF_EXTCLK_gc,  /* External clock on XTALHF1 pin */
    } SELHF;
#endif

#if defined(CLKCTRL_SOURCE_gm)
    /* PLL Source select */
    typedef enum
    {
      SOURCE_OSCHF  = CLKCTRL_SOURCE_OSCHF_gc,   /* Internal High Frequency Oscillator */
#if defined(CLKCTRL_SOURCE_EXTCLK_gc)
      SOURCE_EXTCLK = CLKCTRL_SOURCE_EXTCLK_gc,  /* External Clock */
#endif
#if defined(CLKCTRL_SOURCE_XOSCHF_gc)
      SOURCE_XOSCHF = CLKCTRL_SOURCE_XOSCHF_gc,  /* High frequency external clock or external high frequency oscillator as PLL source */
#endif
    } SOURCE;
#endif

#if defined(CLKCTRL_SOURCEDIV_gm)
    /* PLL Source Division select */
    typedef enum
    {
       SOURCEDIV_DIV1 = CLKCTRL_SOURCEDIV_DIV1_gc,  /* Source undivided */
       SOURCEDIV_DIV2 = CLKCTRL_SOURCEDIV_DIV2_gc,  /* Divide source by 2 */
       SOURCEDIV_DIV4 = CLKCTRL_SOURCEDIV_DIV4_gc,  /* Divide source by 4 */
       SOURCEDIV_DIV6 = CLKCTRL_SOURCEDIV_DIV6_gc,  /* Divide source by 6 */
    } SOURCEDIV;
#endif

#if defined(CLKCTRL_MULFAC_gm)
    /* PLL Multiplication Factor select */
    typedef enum
    {
      MULFAC_OFF = CLKCTRL_MULFAC_OFF_gc,  /* PLL Disabled */
  #if defined(CLKCTRL_MULFAC_8X_gc)
      MULFAC_8X  = CLKCTRL_MULFAC_8X_gc,   /* Multiply by 8 */
  #endif
  #if defined(CLKCTRL_MULFAC_16X_gc)
      MULFAC_16X = CLKCTRL_MULFAC_16X_gc,  /* Multiply by 16 */
  #endif
  #if defined(CLKCTRL_MULFAC_2x_gc)
      MULFAC_2X  = CLKCTRL_MULFAC_2x_gc,   /* 2 x multiplication factor */
  #endif
  #if defined(CLKCTRL_MULFAC_3x_gc)
      MULFAC_3X  = CLKCTRL_MULFAC_3x_gc,   /* 3 x multiplication factor */
  #endif
    } MULFAC;
#endif

#if defined(CLKCTRL_CLKDIV_bm)
    /* PLL Output Clock Division select */
    typedef enum
    {
      CLKDIV_NONE = CLKCTRL_CLKDIV_NONE_gc,  /* PLL output clock undivided */
      CLKDIV_DIV2 = CLKCTRL_CLKDIV_DIV2_gc,  /* PLL output clock divided by 2 */
    } CLKDIV;
#endif

    /* CLKCTRL.MCLKSTATUS  bit masks and bit positions */
    typedef enum
    {
#if defined(CLKCTRL_OSCHFCTRLA)
      STATUS_SOSC     = CLKCTRL_SOSC_bm,      /* System Oscillator changing bit mask. */
      STATUS_OSCHFS   = CLKCTRL_OSCHFS_bm,    /* High frequency oscillator status bit mask. */
      STATUS_OSC32KS  = CLKCTRL_OSC32KS_bm,   /* 32KHz oscillator status bit mask. */
      STATUS_XOSC32KS = CLKCTRL_XOSC32KS_bm,  /* 32.768 kHz Crystal Oscillator status bit mask. */
      STATUS_EXTS     = CLKCTRL_EXTS_bm,      /* External Clock status bit mask. */
  #if defined(CLKCTRL_PLLS_bm)
      STATUS_PLLS     = CLKCTRL_PLLS_bm,      /* PLL oscillator status bit mask. */
  #endif
#else
      STATUS_SOSC    = CLKCTRL_SOSC_bm,     /* System Oscillator changing bit mask. */
      STATUS_OSC20MS = CLKCTRL_OSC20MS_bm,  /* 20MHz oscillator status bit mask. */
      STATUS_OSC32KS = CLKCTRL_OSC32KS_bm,  /* 32KHz oscillator status bit mask. */
      STATUS_EXTS    = CLKCTRL_EXTS_bm,     /* External Clock status bit mask. */
#endif
    } STATUS;

    typedef struct
    {
#if defined(CLKCTRL_OSCHFCTRLA)
      uint8_t ctrla;
      uint8_t ctrlb;
      uint8_t oschf;
#else
      uint8_t ctrla;
      uint8_t ctrlb;
      uint8_t mclklock;
      uint8_t osc20mcaliba;
      uint8_t osc20mcalibb;
#endif
    } SAVE;

    static void setup(void)
    {
#if defined(CLKCTRL_FRQSEL_gm)
      setup(FRQSEL_24M);
#elif defined(CLKCTRL_OSCHFCTRLA)
      setup(CLKSEL_OSCHF);
#else
      setup(CLKSEL_OSC20M);
#endif
    }

#if defined(CLKCTRL_FRQSEL_gm)
    static void setup(FRQSEL frqsel)
    {
      oscfrq(frqsel);
      setup(CLKSEL_OSCHF);
    }
#endif

    static void setup(CLKSEL clksel, PDIV pdiv = PDIV_DIV1, uint32_t extclk = CONFIG_EXTCLK)
    {
      _extclock  = extclk;
      _frequency = 0;
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLA, clksel);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB, pdiv);
    }

#if defined(CLKCTRL_FRQSEL_gm)
    static void oscfrq(FRQSEL frqsel = FRQSEL_24M, AUTOTUNE autotune = AUTOTUNE_OFF)
    {
      _frequency = 0;
  #if defined(CLKCTRL_AUTOTUNE_gm)
      _PROTECTED_WRITE(CLKCTRL.OSCHFCTRLA, (CLKCTRL.OSCHFCTRLA & ~(CLKCTRL_FRQSEL_gm | CLKCTRL_AUTOTUNE_gm)) | frqsel | autotune);
  #else
      _PROTECTED_WRITE(CLKCTRL.OSCHFCTRLA, (CLKCTRL.OSCHFCTRLA & ~(CLKCTRL_FRQSEL_gm | CLKCTRL_AUTOTUNE_bm)) | frqsel | (autotune ? CLKCTRL_AUTOTUNE_bm : 0));
  #endif
      while (!(CLKCTRL.MCLKSTATUS & CLKCTRL_OSCHFS_bm))
        continue;
    }
#endif

#if defined(CLKCTRL_OSCHFTUNE)
    static void osctune(uint8_t value)
    {
      CLKCTRL.OSCHFTUNE = value;
    }
#endif

    static void output(bool enable = true)
    {
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLA, (CLKCTRL.MCLKCTRLA & ~CLKCTRL_CLKOUT_bm) | (enable ? CLKCTRL_CLKOUT_bm : 0));
    }

#if defined(CLKCTRL_MCLKTIMEBASE)
    static void timebase(uint8_t value)
    {
      CLKCTRL.MCLKTIMEBASE = value;
    }
#endif

#if defined(CLKCTRL_SOURCE_gm)
    static void pll(SOURCE src = SOURCE_OSCHF, MULFAC mulfac = MULFAC_OFF)
    {
      _PROTECTED_WRITE(CLKCTRL.PLLCTRLA, (CLKCTRL.PLLCTRLA & ~(CLKCTRL_SOURCE_gm | CLKCTRL_MULFAC_gm)) | src | mulfac);
    }
#endif

#if defined(CLKCTRL_SOURCEDIV_gm)
    static void pll(SOURCEDIV srcdiv = SOURCEDIV_DIV1)
    {
      _PROTECTED_WRITE(CLKCTRL.PLLCTRLA, (CLKCTRL.PLLCTRLA & ~CLKCTRL_SOURCEDIV_gm) | srcdiv);
    }
#endif

#if defined(CLKCTRL_CLKDIV_bm)
    static void pll(CLKDIV clkdiv = CLKDIV_NONE)
    {
      _PROTECTED_WRITE(CLKCTRL.PLLCTRLB, clkdiv);
    }
#endif

#if defined(CLKCTRL_XOSC32KCTRLA)
  #if defined(CLKCTRL_LPMODE_bm)
    static void xosc32k(bool enable, CSUT csut, bool sel, bool lpmode)
    {
      _PROTECTED_WRITE(CLKCTRL.XOSC32KCTRLA, (CLKCTRL.XOSC32KCTRLA & ~(CLKCTRL_CSUT_gm | CLKCTRL_SEL_bm | CLKCTRL_LPMODE_bm | CLKCTRL_ENABLE_bm)) | csut | (sel ? CLKCTRL_SEL_bm : 0) | (lpmode ? CLKCTRL_LPMODE_bm : 0) | (enable ? CLKCTRL_ENABLE_bm : 0));
    }
  #else
    static void xosc32k(bool enable, CSUT csut, bool sel)
    {
      _PROTECTED_WRITE(CLKCTRL.XOSC32KCTRLA, (CLKCTRL.XOSC32KCTRLA & ~(CLKCTRL_CSUT_gm | CLKCTRL_SEL_bm | CLKCTRL_ENABLE_bm)) | csut | (sel ? CLKCTRL_SEL_bm : 0) | (enable ? CLKCTRL_ENABLE_bm : 0));
    }
  #endif
#endif

#if defined(CLKCTRL_XOSCHFCTRLA)
  #if defined(CLKCTRL_FRQRANGE_gm)
    static void xoschf(bool enable, CSUTHF csut, bool sel, FRQRANGE frqrange)
    {
      _PROTECTED_WRITE(CLKCTRL.XOSCHFCTRLA, (CLKCTRL.XOSCHFCTRLA & ~(CLKCTRL_CSUTHF_gm | CLKCTRL_FRQRANGE_gm | CLKCTRL_SELHF_bm | CLKCTRL_ENABLE_bm)) | csut | frqrange | (sel ? CLKCTRL_SEL_bm : 0) | (enable ? CLKCTRL_ENABLE_bm : 0));
    }
  #else
    static void xoschf(bool enable, CSUTHF csut, bool sel)
    {
      _PROTECTED_WRITE(CLKCTRL.XOSCHFCTRLA, (CLKCTRL.XOSCHFCTRLA & ~(CLKCTRL_CSUTHF_gm | CLKCTRL_SELHF_bm | CLKCTRL_ENABLE_bm)) | csut | (sel ? CLKCTRL_SEL_bm : 0) | (enable ? CLKCTRL_ENABLE_bm : 0));
    }
  #endif
#endif

#if defined(CLKCTRL_OSCHFCTRLA)

    static uint32_t frequency(void)
    {
      if (_frequency == 0)
      {
        static const uint8_t  PDIV[] = { 2, 4, 8, 16, 32, 64, 1, 1, 6, 10, 12, 24, 48 };
        uint32_t clock = 0;
        switch (CLKCTRL.MCLKCTRLA & CLKCTRL_CLKSEL_gm)
        {
          case CLKSEL_OSCHF  : clock = Fuse::OSCCFG::frequency(); break;
          case CLKSEL_OSC32K :
          case CLKSEL_XOSC32K: clock = 32768; break;
          case CLKSEL_EXTCLK : clock = _extclock; break;
        }
        _frequency = CLKCTRL.MCLKCTRLB & CLKCTRL_PEN_bm ? clock / PDIV[(CLKCTRL.MCLKCTRLB & CLKCTRL_PDIV_gm) >> CLKCTRL_PDIV_gp] : clock;
      }
      return _frequency;
    }

    static void runstdby(CLKSEL clksel, bool enable)
    {
      switch (clksel)
      {
        case CLKSEL_OSCHF:
          _PROTECTED_WRITE(CLKCTRL.OSCHFCTRLA, (CLKCTRL.OSCHFCTRLA & ~CLKCTRL_RUNSTDBY_bm) | (enable ? CLKCTRL_RUNSTDBY_bm : 0));
          break;
        case CLKSEL_OSC32K:
          _PROTECTED_WRITE(CLKCTRL.OSC32KCTRLA, (CLKCTRL.OSC32KCTRLA & ~CLKCTRL_RUNSTDBY_bm) | (enable ? CLKCTRL_RUNSTDBY_bm : 0));
          break;
        case CLKSEL_XOSC32K:
          _PROTECTED_WRITE(CLKCTRL.XOSC32KCTRLA, (CLKCTRL.XOSC32KCTRLA & ~CLKCTRL_RUNSTDBY_bm) | (enable ? CLKCTRL_RUNSTDBY_bm : 0));
          break;
  #if defined(CLKCTRL_XOSCHFCTRLA)
        case CLKSEL_EXTCLK:
          _PROTECTED_WRITE(CLKCTRL.XOSCHFCTRLA, (CLKCTRL.XOSCHFCTRLA & ~CLKCTRL_RUNSTDBY_bm) | (enable ? CLKCTRL_RUNSTDBY_bm : 0));
          break;
  #endif
  #if defined(CLKCTRL_CLKSEL_PLL_gc)
        case CLKSEL_PLL:
          _PROTECTED_WRITE(CLKCTRL.PLLCTRLA, (CLKCTRL.PLLCTRLA & ~CLKCTRL_RUNSTDBY_bm) | (enable ? CLKCTRL_RUNSTDBY_bm : 0));
          break;
  #endif
        default:
          break;
      }
    }

    static void save(SAVE &buf)
    {
      buf.ctrla = CLKCTRL.MCLKCTRLA;
      buf.ctrlb = CLKCTRL.MCLKCTRLB;
      buf.oschf = CLKCTRL.OSCHFCTRLA;
    }

    static void restore(SAVE &buf)
    {
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB , 0);
      _PROTECTED_WRITE(CLKCTRL.OSCHFCTRLA, buf.oschf);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLA , buf.ctrla);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB , buf.ctrlb);
      _frequency = 0;
    }

#else

    static uint32_t frequency(void)
    {
      if (_frequency == 0)
      {
        static const uint8_t PDIV[] = { 2, 4, 8, 16, 32, 64, 1, 1, 6, 10, 12, 24, 48 };
        uint32_t clock = 0;
        switch (CLKCTRL.MCLKCTRLA & CLKCTRL_CLKSEL_gm)
        {
          case CLKSEL_OSC20M   : clock = Fuse::OSCCFG::frequency(); break;
          case CLKSEL_OSCULP32K: clock = 32768; break;
          case CLKSEL_EXTCLK   : clock = _extclock; break;
        }
        _frequency = CLKCTRL.MCLKCTRLB & CLKCTRL_PEN_bm ? clock / PDIV[(CLKCTRL.MCLKCTRLB & CLKCTRL_PDIV_gm) >> CLKCTRL_PDIV_gp] : clock;
      }
      return _frequency;
    }

    static void runstdby(CLKSEL clksel, bool enable)
    {
      switch (clksel)
      {
        case CLKSEL_OSC20M:
          if (enable)
            _PROTECTED_WRITE(CLKCTRL.OSC20MCTRLA, CLKCTRL.OSC20MCTRLA | CLKCTRL_RUNSTDBY_bm);
          else
            _PROTECTED_WRITE(CLKCTRL.OSC20MCTRLA, CLKCTRL.OSC20MCTRLA & ~CLKCTRL_RUNSTDBY_bm);
          break;
        case CLKSEL_OSCULP32K:
          if (enable)
            _PROTECTED_WRITE(CLKCTRL.OSC32KCTRLA, CLKCTRL.OSC32KCTRLA | CLKCTRL_RUNSTDBY_bm);
          else
            _PROTECTED_WRITE(CLKCTRL.OSC32KCTRLA, CLKCTRL.OSC32KCTRLA & ~CLKCTRL_RUNSTDBY_bm);
          break;
        default:
          break;
      }
    }

    static void lock(bool enable)
    {
      _PROTECTED_WRITE(CLKCTRL.MCLKLOCK, enable ? CLKCTRL_LOCKEN_bm : 0);
    }

    static void calibFreq(uint8_t value)
    {
      _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBA, value);
    }

    static uint8_t calibFeq(void)
    {
      return CLKCTRL.OSC20MCALIBA;
    }

    static void calibTemp(uint8_t value)
    {
      _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBB, value);
    }

    static uint8_t calibTemp(void)
    {
      return CLKCTRL_OSC20MCALIBB;
    }

    static void calibLock(bool enable)
    {
      if (enable)
        _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBB, CLKCTRL.OSC20MCALIBB | CLKCTRL_LOCK_bm);
      else
        _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBB, CLKCTRL.OSC20MCALIBB & ~CLKCTRL_LOCK_bm);
    }

    static void save(SAVE &buf)
    {
      buf.ctrla        = CLKCTRL.MCLKCTRLA;
      buf.ctrlb        = CLKCTRL.MCLKCTRLB;
      buf.mclklock     = CLKCTRL.MCLKLOCK;
      buf.osc20mcaliba = CLKCTRL.OSC20MCALIBA;
      buf.osc20mcalibb = CLKCTRL.OSC20MCALIBB;
    }

    static void restore(SAVE &buf)
    {
      _PROTECTED_WRITE(CLKCTRL.MCLKLOCK    , 0);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB   , 0);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLA   , buf.ctrla);
      _PROTECTED_WRITE(CLKCTRL.MCLKCTRLB   , buf.ctrlb);
      _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBA, buf.osc20mcaliba);
      _PROTECTED_WRITE(CLKCTRL.OSC20MCALIBB, buf.osc20mcalibb);
      _PROTECTED_WRITE(CLKCTRL.MCLKLOCK    , buf.mclklock);
      _frequency = 0;
    }

#endif

    static STATUS status(void)
    {
      return (STATUS)CLKCTRL.MCLKSTATUS;
    }

  protected:

    static uint32_t _extclock;
    static uint32_t _frequency;
};