/*
  avr8_dac.h - DAC Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_vref.h"

#if defined(DAC_ENABLE_bm)

#if !defined(DAC_DATA_gm)
  #define DAC_DATA_gm 0xFF
  #define DAC_DATA_gp 0
#endif

class Dac
{
  public:

#if defined(DAC_OUTRANGE_gm)
    /* Output Buffer Range select */
    typedef enum
    {
      OUTRANGE_AUTO    = DAC_OUTRANGE_AUTO_gc,  /* Output buffer automatically choose best range */
      OUTRANGE_LOW     = DAC_OUTRANGE_LOW_gc,   /* Output buffer configured to low range */
      OUTRANGE_HIGH    = DAC_OUTRANGE_HIGH_gc,  /* Output buffer configured to high range */
      OUTRANGE_DEFAULT = OUTRANGE_AUTO,
    } OUTRANGE;
#endif

    Dac(DAC_t& dac) : _dac(&dac), _ch((&dac - &DAC0) / sizeof(dac))
    {
    }

    /* virtual */ ~Dac(void)
    {
      end();
    }

    void begin(Vref::DACREF refsel, bool alwayson = false)
    {
      end();
      Vref::dac(_ch, refsel, alwayson);
      _dac->DATA  = 0;
      _dac->CTRLA = DAC_ENABLE_bm;
    }

    void end(void)
    {
      _dac->CTRLA = 0;
    }

    void runstdby(bool enable = true)
    {
      _dac->CTRLA = (_dac->CTRLA & ~DAC_RUNSTDBY_bm) | (enable ? DAC_RUNSTDBY_bm : 0);
    }

#if defined(DAC_OUTRANGE_gm)
    void output(bool enable = true, OUTRANGE outrange = OUTRANGE_AUTO)
    {
      _dac->CTRLA = (_dac->CTRLA & ~(DAC_OUTEN_bm | DAC_OUTRANGE_gm)) | (enable ? DAC_OUTEN_bm : 0) | outrange;
    }
#else
    void output(bool enable = true)
    {
      _dac->CTRLA = (_dac->CTRLA & ~DAC_OUTEN_bm) | (enable ? DAC_OUTEN_bm : 0);
    }
#endif

    void data(uint16_t value)
    {
      _dac->DATA = value;
    }

    uint16_t data(void)
    {
      return _dac->DATA;
    }

    uint16_t mask(void)
    {
      return DAC_DATA_gm;
    }

  protected:

    DAC_t  *_dac;
    uint8_t _ch;
};

#endif
