/*
  avr8_fuse.h - FUSH/SIGROW/USERROW Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include <string.h>
#include "avr8_config.h"
#include "avr8_fuse_dfp.h"

#if defined(RSTPINCFG_RESET_gc)
  #define RSTPINCFG_RST_gc RSTPINCFG_RESET_gc
#endif

class Fuse
{
  public:

    /* Watchdog Configuration */
    class WDTCFG
    {
      public:

        /* Watchdog Window Timeout Period select */
        typedef enum
        {
          WINDOW_OFF    = WINDOW_OFF_gc,     /* Window mode off */
          WINDOW_8CLK   = WINDOW_8CLK_gc,    /* 8 cycles (8ms) */
          WINDOW_16CLK  = WINDOW_16CLK_gc,   /* 16 cycles (16ms) */
          WINDOW_32CLK  = WINDOW_32CLK_gc,   /* 32 cycles (32ms) */
          WINDOW_64CLK  = WINDOW_64CLK_gc,   /* 64 cycles (64ms) */
          WINDOW_128CLK = WINDOW_128CLK_gc,  /* 128 cycles (0.128s) */
          WINDOW_256CLK = WINDOW_256CLK_gc,  /* 256 cycles (0.256s) */
          WINDOW_512CLK = WINDOW_512CLK_gc,  /* 512 cycles (0.512s) */
          WINDOW_1KCLK  = WINDOW_1KCLK_gc,   /* 1K cycles (1.0s) */
          WINDOW_2KCLK  = WINDOW_2KCLK_gc,   /* 2K cycles (2.0s) */
          WINDOW_4KCLK  = WINDOW_4KCLK_gc,   /* 4K cycles (4.1s) */
          WINDOW_8KCLK  = WINDOW_8KCLK_gc,   /* 8K cycles (8.2s) */
        } WINDOW;

        /* Watchdog Timeout Period select */
        typedef enum
        {
          PERIOD_OFF    = PERIOD_OFF_gc,     /* Watch-Dog timer Off */
          PERIOD_8CLK   = PERIOD_8CLK_gc,    /* 8 cycles (8ms) */
          PERIOD_16CLK  = PERIOD_16CLK_gc,   /* 16 cycles (16ms) */
          PERIOD_32CLK  = PERIOD_32CLK_gc,   /* 32 cycles (32ms) */
          PERIOD_64CLK  = PERIOD_64CLK_gc,   /* 64 cycles (64ms) */
          PERIOD_128CLK = PERIOD_128CLK_gc,  /* 128 cycles (0.128s) */
          PERIOD_256CLK = PERIOD_256CLK_gc,  /* 256 cycles (0.256s) */
          PERIOD_512CLK = PERIOD_512CLK_gc,  /* 512 cycles (0.512s) */
          PERIOD_1KCLK  = PERIOD_1KCLK_gc,   /* 1K cycles (1.0s) */
          PERIOD_2KCLK  = PERIOD_2KCLK_gc,   /* 2K cycles (2.0s) */
          PERIOD_4KCLK  = PERIOD_4KCLK_gc,   /* 4K cycles (4.1s) */
          PERIOD_8KCLK  = PERIOD_8KCLK_gc,   /* 8K cycles (8.2s) */
        } PERIOD;

        static WINDOW window(void)
        {
          return (WINDOW)(FUSE.WDTCFG & FUSE_WINDOW_gm);
        }

        static PERIOD period(void)
        {
          return (PERIOD)(FUSE.WDTCFG & FUSE_PERIOD_gm);
        }
    };

    /* BOD Configuration */
    class BODCFG
    {
      public:

        /* BOD Level select */
        typedef enum
        {
#if defined(BOD_LVL_BODLEVEL0_gc)
          LVL_BODLEVEL0 = BOD_LVL_BODLEVEL0_gc,  /* 1.8 V */
#endif
#if defined(BOD_LVL_BODLEVEL1_gc)
          LVL_BODLEVEL1 = BOD_LVL_BODLEVEL1_gc,  /* 2.1 V */
#endif
#if defined(BOD_LVL_BODLEVEL2_gc)
          LVL_BODLEVEL2 = BOD_LVL_BODLEVEL2_gc,  /* 2.6 V */
#endif
#if defined(BOD_LVL_BODLEVEL3_gc)
          LVL_BODLEVEL3 = BOD_LVL_BODLEVEL3_gc,  /* 2.9 V */
#endif
#if defined(BOD_LVL_BODLEVEL4_gc)
          LVL_BODLEVEL4 = BOD_LVL_BODLEVEL4_gc,  /* 3.3 V */
#endif
#if defined(BOD_LVL_BODLEVEL5_gc)
          LVL_BODLEVEL5 = BOD_LVL_BODLEVEL5_gc,  /* 3.7 V */
#endif
#if defined(BOD_LVL_BODLEVEL6_gc)
          LVL_BODLEVEL6 = BOD_LVL_BODLEVEL6_gc,  /* 4.0 V */
#endif
#if defined(BOD_LVL_BODLEVEL7_gc)
          LVL_BODLEVEL7 = BOD_LVL_BODLEVEL7_gc,  /* 4.2 V */
#endif
        } LVL;

        /* BOD Operation in Active Mode select */
        typedef enum
        {
          ACTIVE_DISABLE = BOD_ACTIVE_DISABLE_gc,     /* BOD disabled */
          ACTIVE_ENABLED = BOD_ACTIVE_ENABLED_gc,     /* BOD enabled in continuous mode */
          ACTIVE_SAMPLED = BOD_ACTIVE_SAMPLED_gc,     /* BOD enabled in sampled mode */
          ACTIVE_ENWAIT  = BOD_ACTIVE_ENWAKE_gc,      /* BOD enabled in continuous mode. Execution is halted at wake-up until BOD is running. */
        } ACTIVE;

        /* BOD Operation in Sleep Mode select */
        typedef enum
        {
          SLEEP_DISABLE = BOD_SLEEP_DISABLE_gc,   /* BOD disabled */
          SLEEP_ENABLED = BOD_SLEEP_ENABLED_gc,   /* BOD enabled in continuous mode */
          SLEEP_SAMPLED = BOD_SLEEP_SAMPLED_gc,   /* BOD enabled in sampled mode */
        } SLEEP;

        /* BOD Sample Frequency select */
        typedef enum
        {
#if defined(CLKCTRL_OSCHFCTRLA)
          SAMPFREQ_128HZ = BOD_SAMPFREQ_128HZ_gc,  /* Sample frequency is 128 Hz */
          SAMPFREQ_32HZ  = BOD_SAMPFREQ_32HZ_gc,   /* Sample frequency is 32 Hz */
#else
          SAMPFREQ_1KHZ  = BOD_SAMPFREQ_1KHZ_gc,   /* 1kHz sampling frequency */
          SAMPFREQ_125HZ = BOD_SAMPFREQ_125HZ_gc,  /* 125Hz sampling frequency */
#endif
        } SAMPFREQ;

        static LVL lvl(void)
        {
          return (LVL)(FUSE.BODCFG & FUSE_LVL_gm);
        }

        static SAMPFREQ samplefreq(void)
        {
          return (SAMPFREQ)(FUSE.BODCFG & FUSE_SAMPFREQ_bm);
        }

        static ACTIVE active(void)
        {
          return (ACTIVE)(FUSE.BODCFG & FUSE_ACTIVE_gm);
        }

        static SLEEP sleep(void)
        {
          return (SLEEP)(FUSE.BODCFG & FUSE_SLEEP_gm);
        }
    };      

    /* Oscillator Configuration */
    class OSCCFG
    {
      public:

#if defined(FUSE_CLKSEL_gm)
        /* Frequency Select */
        typedef enum
        {
          CLKSEL_OSCHF  = CLKSEL_OSCHF_gc,   /* Device running on internal high frequency oscillator */
          CLKSEL_OSC32K = CLKSEL_OSC32K_gc,  /* Device running on internal 32.768 kHz oscillator */
        } CLKSEL;

        static CLKSEL clksel(void)
        {
          return (CLKSEL)(FUSE.OSCCFG & FUSE_CLKSEL_gm);
        }

        static uint32_t frequency(void)
        {
          static const uint32_t HFFREQ[] = { 1000000, 2000000, 3000000, 4000000, 0, 8000000, 12000000, 16000000, 20000000, 24000000 };
          return HFFREQ[(CLKCTRL.OSCHFCTRLA & CLKCTRL_FRQSEL_gm) >> CLKCTRL_FRQSEL_gp];
        }
#elif defined(FUSE_OSCHFFRQ_bm)
        /* High-frequency Oscillator Frequency select */
        typedef enum
        {
          OSCHFFRQ_20M = OSCHFFRQ_20M_gc,   /* OSCHF running at 20 MHz */
          OSCHFFRQ_16M = OSCHFFRQ_16M_gc,   /* OSCHF running at 16 MHz */
        } OSCHFFRQ;

        static OSCHFFRQ freqsel(void)
        {
          return (OSCHFFRQ)(FUSE.OSCCFG & FUSE_OSCHFFRQ_bm);
        }

        static uint32_t frequency(void)
        {
          switch (freqsel())
          {
            case OSCHFFRQ_20M: return 20000000UL;
            case OSCHFFRQ_16M: return 16000000UL;
          }
          return 0;
        }
#else
        /* Frequency Select select */
        typedef enum
        {
          FREQSEL_16M = FREQSEL_16MHZ_gc,  /* 16 MHz */
          FREQSEL_20M = FREQSEL_20MHZ_gc,  /* 20 MHz */
        } FREQSEL;

        static FREQSEL freqsel(void)
        {
          return (FREQSEL)(FUSE.OSCCFG & FUSE_FREQSEL_gm);
        }

        static uint32_t frequency(void)
        {
          switch (freqsel())
          {
            case FREQSEL_16M: return 16000000UL;
            case FREQSEL_20M: return 20000000UL;
          }
          return 0;
        }

        static bool osclock(void)
        {
          return (FUSE.OSCCFG & FUSE_OSCLOCK_bm) != 0;
        }
#endif
    };

    /* System Configuration 0 */
    class SYSCFG0
    {
      public:

        /* CRC Source select */
        typedef enum
        {
          CRCSRC_FLASH   = CRCSRC_FLASH_gc,    /* The CRC is performed on the entire Flash (boot, application code and application data section). */
          CRCSRC_BOOT    = CRCSRC_BOOT_gc,     /* The CRC is performed on the boot section of Flash */
          CRCSRC_BOOTAPP = CRCSRC_BOOTAPP_gc,  /* The CRC is performed on the boot and application code section of Flash */
          CRCSRC_NOCRC   = CRCSRC_NOCRC_gc,    /* Disable CRC. */
        } CRCSRC;

        static CRCSRC crcsrc(void)
        {
          return (CRCSRC)(FUSE.SYSCFG0 & FUSE_CRCSRC_gm);
        }

#if defined(FUSE_CRCSEL_bm)
        /* CRC Select */
        typedef enum CRCSEL_enum
        {
          CRCSEL_CRC16 = CRCSEL_CRC16_gc,  /* Enable CRC16 */
          CRCSEL_CRC32 = CRCSEL_CRC32_gc,  /* Enable CRC32 */
        } CRCSEL;

        static CRCSEL crcsel(void)
        {
          return (CRCSEL)(FUSE.SYSCFG0 & FUSE_CRCSEL_bm);
        }
#endif

        /* Reset Pin Configuration select */
        typedef enum
        {
#if defined(RSTPINCFG_NONE)
          RSTPINCFG_NONE   = RSTPINCFG_NONE_gc,     /* No External Reset */
#endif
#if defined(RSTPINCFG_RST_gc)
          RSTPINCFG_RST    = RSTPINCFG_RST_gc,      /* configured as RESET pin */
#endif
#if defined(RSTPINCFG_GPIO_gc)
          RSTPINCFG_GPIO   = RSTPINCFG_GPIO_gc,     /* GPIO mode */
#endif
#if defined(RSTPINCFG_UPDI_gc)
          RSTPINCFG_UPDI   = RSTPINCFG_UPDI_gc,     /* UPDI mode */
#endif
#if defined(RSTPINCFG_PDIRST_gc)
          RSTPINCFG_PDIRST = RSTPINCFG_PDIRST_gc,  /* PDI on PDI pad, reset on alternative reset pad */
#endif
        } RSTPINCFG;

        static RSTPINCFG rstpincfg(void)
        {
          return (RSTPINCFG)(FUSE.SYSCFG0 & FUSE_RSTPINCFG_gm);
        }

#if defined(FUSE_UPDIPINCFG_bm)
        /* UPDI Pin Configuration select */
        typedef enum UPDIPINCFG_enum
        {
          UPDIPINCFG_GPIO = UPDIPINCFG_GPIO_gc,  /* Configured as GPIO pin */
          UPDIPINCFG_UPDI = UPDIPINCFG_UPDI_gc,  /* Configured as UPDI pin */
        } UPDIPINCFG;

        static UPDIPINCFG updipincfg(void)
        {
          return (UPDIPINCFG)(FUSE.SYSCFG0 & FUSE_UPDIPINCFG_bm);
        }
#endif

#if defined(FUSE_EESAVE_bm)
        /* EEPROM Save select */
        typedef enum
        {
          EESAVE_DISABLE = 0 << FUSE_EESAVE_bp,  /* EEPROM content is erased during chip erase */
          EESAVE_ENABLE  = 1 << FUSE_EESAVE_bp,  /* EEPROM content is preserved during chip erase */
        } EESAVE;

        static EESAVE eesave(void)
        {
          return (EESAVE)(FUSE.SYSCFG0 & FUSE_EESAVE_bm);
        }
#endif
    };

    /* System Configuration 1 */
    class SYSCFG1
    {
      public:

        /* Startup Time select */
        typedef enum
        {
          SUT_0MS  = SUT_0MS_gc,   /* 0 ms */
          SUT_1MS  = SUT_1MS_gc,   /* 1 ms */
          SUT_2MS  = SUT_2MS_gc,   /* 2 ms */
          SUT_4MS  = SUT_4MS_gc,   /* 4 ms */
          SUT_8MS  = SUT_8MS_gc,   /* 8 ms */
          SUT_16MS = SUT_16MS_gc,  /* 16 ms */
          SUT_32MS = SUT_32MS_gc,  /* 32 ms */
          SUT_64MS = SUT_64MS_gc,  /* 64 ms */
        } SUT;

        static SUT sut(void)
        {
          return (SUT)(FUSE.SYSCFG1 & FUSE_SUT_gm);
        }

#if defined(FUSE_MVSYSCFG_gm)
        /* MVIO System Configuration select */
        typedef enum
        {
          MVSYSCFG_DUAL   = MVSYSCFG_DUAL_gc,    /* Device used in a dual supply configuration */
          MVSYSCFG_SINGLE = MVSYSCFG_SINGLE_gc,  /* Device used in a single supply configuration */
        } MVSYSCFG;

        static MVSYSCFG mvsyscfg(void)
        {
          return (MVSYSCFG)(FUSE.SYSCFG1 & FUSE_MVSYSCFG_gm);
        }
#endif

#if defined(FUSE_USBSINK_bm)
        /* USB Voltage Regulator Current Sink Enable select */
        typedef enum
        {
          USBSINK_DISABLE = USBSINK_DISABLE_gc,  /* USB VREG can not sink current */
          USBSINK_ENABLE  = USBSINK_ENABLE_gc,   /* USB VREG can sink current */
        } USBSINK;

        static USBSINK usbsink(void)
        {
          return (USBSINK)(FUSE.SYSCFG1 & FUSE_USBSINK_bm);
        }
#endif
    };

#if defined(FUSE_PDICFG)
    /* Programming and Debugging Interface Configuration */
    class PDICFG
    {
      public:

        /* Protection Level select */
        typedef enum LEVEL_enum
        {
          LEVEL_NVMACCDIS = LEVEL_NVMACCDIS_gc,  /* NVM Access through UPDI disabled */
          LEVEL_BASIC     = LEVEL_BASIC_gc,      /* UPDI and UPDI pins working normally */
        } LEVEL;

        static LEVEL level(void)
        {
          return (LEVEL)(FUSE.PDICFG & FUSE_LEVEL_gm);
        }

        /* NVM Protection Activation Key select */
        typedef enum KEY_enum
        {
          KEY_NOTACT = KEY_NOTACT_gc,  /* Not Active */
          KEY_NVMACT = KEY_NVMACT_gc,  /* NVM Protection Active */
        } KEY;

        static KEY key(void)
        {
          return (KEY)(FUSE.PDICFG & FUSE_KEY_gm);
        }
    };
#endif

#if defined(FUSE_CODESIZE)
    /* Application Code Section End */
    class CODESIZE
    {
      public:

        static uint8_t size(void)
        {
          return FUSE.CODESIZE;
         }
    };
#endif

#if defined(FUSE_BOOTSIZE)
    /* Application Code Section End */
    class BOOTSIZE
    {
      public:

        static uint8_t size(void)
        {
          return FUSE.BOOTSIZE;
        }
    };
#endif

#if defined(FUSE_APPEND)
    /* Application Code Section End */
    class APPENDCFG
    {
      public:

        static uint8_t append(void)
        {
          return FUSE.APPEND;
        }
    };
#endif

#if defined(FUSE_BOOTEND)
    /* Boot End */
    class BOOTEND
    {
      public:

        static uint8_t bootend(void)
        {
          return FUSE.BOOTEND;
        }
    };
#endif

#if defined(LOCK_BIT)
    /* Lock Bits */
    class LOCKBIT
    {
      public:

        /* Lock Bits select */
        typedef enum
        {
          LB_RWLOCK = LB_RWLOCK_gc,  /* Read and write lock */
          LB_NOLOCK = LB_NOLOCK_gc,  /* No locks */
        } LB;

        static LB lockbit(void)
        {
          return (LB)(LOCK_BIT.LOCKBIT & LOCKBIT_LB_gm);
        }
    };
#endif

};

class Sigrow
{
  public:

    static uint8_t deviceid(uint8_t index)
    {
      return index < deviceid() ? ((register8_t*)(&SIGROW.DEVICEID0))[index] : 0;
    }

    static uint8_t deviceid(void)
    {
      return 3;
    }

    static uint8_t sernum(uint8_t index)
    {
      return index < sernum() ? ((register8_t*)(&SIGROW.SERNUM0))[index] : 0;
    }

    static uint8_t sernum(void)
    {
#if defined(SIGROW_SERNUM15)
      return 16;
#else
      return 10;
#endif
    }

    /* Temperature Sensor Calibration 0 */
    static uint16_t tempsense0(void)
    {
      return SIGROW.TEMPSENSE0;
    }

    /* Temperature Sensor Calibration 1 */
    static uint16_t tempsense1(void)
    {
      return SIGROW.TEMPSENSE1;
    }

#if defined(SIGROW_OSC20ERR3V)
    /* OSC error */
    static int8_t oscerr(uint16_t vdd = CONFIG_VDD)
    {
      switch (Fuse::OSCCFG::freqsel())
      {
        case Fuse::OSCCFG::FREQSEL_16M: return CONFIG_VDD < 4000 ? SIGROW.OSC16ERR3V : SIGROW.OSC16ERR5V;
        case Fuse::OSCCFG::FREQSEL_20M: return CONFIG_VDD < 4000 ? SIGROW.OSC20ERR3V : SIGROW.OSC20ERR5V;
      }
      return 0;
    }
#endif
};

class Userrow
{
  public:

    static uint8_t read(uint8_t index)
    {
      return index < size() ? ((register8_t*)&USERROW)[index] : 0;
    }

    static void write(uint8_t index, uint8_t value)
    {
      if (index < size())
      {
        eeprom_busy_wait();
        eeprom_update_byte((uint8_t*)((&USERROW - MAPPED_EEPROM_START) + index), value);
      }        
    }

    static size_t size(void)
    {
      return sizeof(USERROW);
    }
};