/*
  avr8_port.h - PORT Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"

#if   defined(PORTG)
  #define PORT_MAX_PORTS 7
#elif defined(PORTF)
  #define PORT_MAX_PORTS 6
#elif defined(PORTE)
  #define PORT_MAX_PORTS 5
#elif defined(PORTD)
  #define PORT_MAX_PORTS 4
#elif defined(PORTC)
  #define PORT_MAX_PORTS 3
#elif defined(PORTB)
  #define PORT_MAX_PORTS 2
#elif defined(PORTA)
  #define PORT_MAX_PORTS 1
#else
  #define PORT_MAX_PORTS 0
#endif

class Port
{
  public:

    /* Port ID */
    typedef enum
    {
#if defined(PORTA)
      PA = 0,
#endif
#if defined(PORTB)
      PB = 1,
#endif
#if defined(PORTC)
      PC = 2,
#endif
#if defined(PORTD)
      PD = 3,
#endif
#if defined(PORTE)
      PE = 4,
#endif
#if defined(PORTF)
      PF = 5,
#endif
#if defined(PORTG)
      PG = 6,
#endif
    } PORT;

    /* Pin ID (pindef_t) */
    typedef enum
    {
#if defined(PORTA)
      PA0 = PA + 0x0100,
      PA1 = PA + 0x0210,
      PA2 = PA + 0x0420,
      PA3 = PA + 0x0830,
      PA4 = PA + 0x1040,
      PA5 = PA + 0x2050,
      PA6 = PA + 0x4060,
      PA7 = PA + 0x8070,
#endif
#if defined(PORTB)
      PB0 = PB + (PA0 - PA),
      PB1 = PB + (PA1 - PA),
      PB2 = PB + (PA2 - PA),
      PB3 = PB + (PA3 - PA),
      PB4 = PB + (PA4 - PA),
      PB5 = PB + (PA5 - PA),
      PB6 = PB + (PA6 - PA),
      PB7 = PB + (PA7 - PA),
#endif
#if defined(PORTC)
      PC0 = PC + (PA0 - PA),
      PC1 = PC + (PA1 - PA),
      PC2 = PC + (PA2 - PA),
      PC3 = PC + (PA3 - PA),
      PC4 = PC + (PA4 - PA),
      PC5 = PC + (PA5 - PA),
      PC6 = PC + (PA6 - PA),
      PC7 = PC + (PA7 - PA),
#endif
#if defined(PORTD)
      PD0 = PD + (PA0 - PA),
      PD1 = PD + (PA1 - PA),
      PD2 = PD + (PA2 - PA),
      PD3 = PD + (PA3 - PA),
      PD4 = PD + (PA4 - PA),
      PD5 = PD + (PA5 - PA),
      PD6 = PD + (PA6 - PA),
      PD7 = PD + (PA7 - PA),
#endif
#if defined(PORTE)
      PE0 = PE + (PA0 - PA),
      PE1 = PE + (PA1 - PA),
      PE2 = PE + (PA2 - PA),
      PE3 = PE + (PA3 - PA),
      PE4 = PE + (PA4 - PA),
      PE5 = PE + (PA5 - PA),
      PE6 = PE + (PA6 - PA),
      PE7 = PE + (PA7 - PA),
#endif
#if defined(PORTF)
      PF0 = PF + (PA0 - PA),
      PF1 = PF + (PA1 - PA),
      PF2 = PF + (PA2 - PA),
      PF3 = PF + (PA3 - PA),
      PF4 = PF + (PA4 - PA),
      PF5 = PF + (PA5 - PA),
      PF6 = PF + (PA6 - PA),
      PF7 = PF + (PA7 - PA),
#endif
#if defined(PORTG)
      PG0 = PG + (PA0 - PA),
      PG1 = PG + (PA1 - PA),
      PG2 = PG + (PA2 - PA),
      PG3 = PG + (PA3 - PA),
      PG4 = PG + (PA4 - PA),
      PG5 = PG + (PA5 - PA),
      PG6 = PG + (PA6 - PA),
      PG7 = PG + (PA7 - PA),
#endif
    } PIN;

    /* Pin Mode */
    typedef enum
    {
      DISABLE,
      INPUT,
      INPUT_PULLUP,
      OUTPUT,
      OPEN_DRAIN,
      OPEN_SOURCE,
    } MODE;

    /* Pin Value */
    typedef enum
    {
      LOW,
      HIGH,
      TOGGLE = 0xFF,
      IGNORE = TOGGLE,
    } VAL;

    /* Input/Sense Configuration select */
    typedef enum
    {
      ISC_INTDISABLE    = PORT_ISC_INTDISABLE_gc,     /* Interrupt disabled but input buffer enabled */
      ISC_BOTHEDGES     = PORT_ISC_BOTHEDGES_gc,      /* Sense Both Edges */
      ISC_RISING        = PORT_ISC_RISING_gc,         /* Sense Rising Edge */
      ISC_FALLING       = PORT_ISC_FALLING_gc,        /* Sense Falling Edge */
      ISC_INPUT_DISABLE = PORT_ISC_INPUT_DISABLE_gc,  /* Digital Input Buffer disabled */
      ISC_LEVEL         = PORT_ISC_LEVEL_gc,          /* Sense low Level */
    } ISC;

    /* Type of Callback Function */
    typedef void (*callback_t)(uint8_t value, uint8_t flags);

    static void disable(void)
    {
      for (uint8_t i = 0; i < PORT_MAX_PORTS; ++i)
      {
        PORT_t *port = getPort(i);
        if (port)
        {
          port->PIN0CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN1CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN2CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN3CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN4CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN5CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN6CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->PIN7CTRL = PORT_ISC_INPUT_DISABLE_gc;
          port->DIR      = 0;
          port->OUT      = 0;
          port->INTFLAGS = 0xFF;
        }
      }
    }

    static void pinMode(PIN pin, MODE mode, VAL value = IGNORE, bool invert = false)
    {
      pindef_t spin = { .port = pin };
      PORT_t* port = getPort(spin.x.port);
      if (port)
      {
        uint8_t ctrl = 0;
        switch (mode)
        {
          case DISABLE:
            ctrl = PORT_ISC_INPUT_DISABLE_gc;
            break;
          case OUTPUT:
            if (value != IGNORE) digitalWrite(pin, value);
            port->DIRSET = spin.x.bmsk;
            break;
          case OPEN_DRAIN:
            digitalWrite(pin, LOW);
            if (value != IGNORE) digitalWriteOD(pin, value);
            ctrl = PORT_PULLUPEN_bm;
            break;
          case OPEN_SOURCE:
            digitalWrite(pin, HIGH);
            if (value == IGNORE) digitalWriteOS(pin, value);
            break;
          case INPUT_PULLUP:
            ctrl = PORT_PULLUPEN_bm;
            /* fall through */
          case INPUT:
            port->DIRCLR = spin.x.bmsk;
            break;
        }
        ((register8_t *)&port->PIN0CTRL)[spin.x.bpos] = ctrl | (invert ? PORT_INVEN_bm : 0);
      }
    }

    /* for pinMode(OUTPUT) */
    static void digitalWrite(PIN pin, VAL value)
    {
      pindef_t spin = { .port = pin };
#if !defined(DEBUG)
      if (__builtin_constant_p(pin))
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) VPORTA.IN |= spin.x.bmsk; else if (value == LOW) VPORTA.OUT &= ~spin.x.bmsk; else VPORTA.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) VPORTB.IN |= spin.x.bmsk; else if (value == LOW) VPORTB.OUT &= ~spin.x.bmsk; else VPORTB.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) VPORTC.IN |= spin.x.bmsk; else if (value == LOW) VPORTC.OUT &= ~spin.x.bmsk; else VPORTC.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) VPORTD.IN |= spin.x.bmsk; else if (value == LOW) VPORTD.OUT &= ~spin.x.bmsk; else VPORTD.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) VPORTE.IN |= spin.x.bmsk; else if (value == LOW) VPORTE.OUT &= ~spin.x.bmsk; else VPORTE.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) VPORTF.IN |= spin.x.bmsk; else if (value == LOW) VPORTF.OUT &= ~spin.x.bmsk; else VPORTF.OUT |= spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) VPORTG.IN |= spin.x.bmsk; else if (value == LOW) VPORTG.OUT &= ~spin.x.bmsk; else VPORTG.OUT |= spin.x.bmsk; break;
#endif
        }
      }
      else
#endif
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) PORTA.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTA.OUTCLR = spin.x.bmsk; else PORTA.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) PORTB.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTB.OUTCLR = spin.x.bmsk; else PORTB.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) PORTC.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTC.OUTCLR = spin.x.bmsk; else PORTC.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) PORTD.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTD.OUTCLR = spin.x.bmsk; else PORTD.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) PORTE.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTE.OUTCLR = spin.x.bmsk; else PORTE.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) PORTF.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTF.OUTCLR = spin.x.bmsk; else PORTF.OUTSET = spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) PORTG.OUTTGL = spin.x.bmsk; else if (value == LOW) PORTG.OUTCLR = spin.x.bmsk; else PORTG.OUTSET = spin.x.bmsk; break;
#endif
        }
      }
    }

    /* for pinMode(OPEN_DRAIN) */
    static void digitalWriteOD(PIN pin, VAL value)
    {
      pindef_t spin = { .port = pin };
#if !defined(DEBUG)
      if (__builtin_constant_p(pin))
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) PORTA.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTA.DIR |= spin.x.bmsk; else VPORTA.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) PORTB.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTB.DIR |= spin.x.bmsk; else VPORTB.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) PORTC.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTC.DIR |= spin.x.bmsk; else VPORTC.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) PORTD.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTD.DIR |= spin.x.bmsk; else VPORTD.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) PORTE.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTE.DIR |= spin.x.bmsk; else VPORTE.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) PORTF.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTF.DIR |= spin.x.bmsk; else VPORTF.DIR &= ~spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) PORTG.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTG.DIR |= spin.x.bmsk; else VPORTG.DIR &= ~spin.x.bmsk; break;
#endif
        }
      }
      else
#endif
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) PORTA.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTA.DIRSET = spin.x.bmsk; else PORTA.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) PORTB.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTB.DIRSET = spin.x.bmsk; else PORTB.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) PORTC.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTC.DIRSET = spin.x.bmsk; else PORTC.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) PORTD.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTD.DIRSET = spin.x.bmsk; else PORTD.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) PORTE.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTE.DIRSET = spin.x.bmsk; else PORTE.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) PORTF.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTF.DIRSET = spin.x.bmsk; else PORTF.DIRCLR = spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) PORTG.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTG.DIRSET = spin.x.bmsk; else PORTG.DIRCLR = spin.x.bmsk; break;
#endif
        }
      }        
    }

    /* for pinMode(OPEN_SOURCE) */
    static void digitalWriteOS(PIN pin, VAL value)
    {
      pindef_t spin = { .port = pin };
#if !defined(DEBUG)
      if (__builtin_constant_p(pin))
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) PORTA.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTA.DIR &= ~spin.x.bmsk; else VPORTA.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) PORTB.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTB.DIR &= ~spin.x.bmsk; else VPORTB.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) PORTC.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTC.DIR &= ~spin.x.bmsk; else VPORTC.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) PORTD.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTD.DIR &= ~spin.x.bmsk; else VPORTD.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) PORTE.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTE.DIR &= ~spin.x.bmsk; else VPORTE.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) PORTF.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTF.DIR &= ~spin.x.bmsk; else VPORTF.DIR |= spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) PORTG.DIRTGL = spin.x.bmsk; else if (value == LOW) VPORTG.DIR &= ~spin.x.bmsk; else VPORTG.DIR |= spin.x.bmsk; break;
#endif
        }
      }
      else
#endif
      {
        switch (spin.x.port)
        {
#if defined(PORTA)
          case 0: if (value == TOGGLE) PORTA.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTA.DIRCLR = spin.x.bmsk; else PORTA.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTB)
          case 1: if (value == TOGGLE) PORTB.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTB.DIRCLR = spin.x.bmsk; else PORTB.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTC)
          case 2: if (value == TOGGLE) PORTC.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTC.DIRCLR = spin.x.bmsk; else PORTC.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTD)
          case 3: if (value == TOGGLE) PORTD.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTD.DIRCLR = spin.x.bmsk; else PORTD.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTE)
          case 4: if (value == TOGGLE) PORTE.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTE.DIRCLR = spin.x.bmsk; else PORTE.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTF)
          case 5: if (value == TOGGLE) PORTF.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTF.DIRCLR = spin.x.bmsk; else PORTF.DIRSET = spin.x.bmsk; break;
#endif
#if defined(PORTG)
          case 6: if (value == TOGGLE) PORTG.DIRTGL = spin.x.bmsk; else if (value == LOW) PORTG.DIRCLR = spin.x.bmsk; else PORTG.DIRSET = spin.x.bmsk; break;
#endif
        }
      }        
    }

    static VAL digitalRead(PIN pin)
    {
      pindef_t spin = { .port = pin };
      switch (spin.x.port)
      {
#if defined(PORTA)
        case 0: return VPORTA.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTB)
        case 1: return VPORTB.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTC)
        case 2: return VPORTC.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTD)
        case 3: return VPORTD.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTE)
        case 4: return VPORTE.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTF)
        case 5: return VPORTF.IN & spin.x.bmsk ? HIGH : LOW;
#endif
#if defined(PORTG)
        case 6: return VPORTG.IN & spin.x.bmsk ? HIGH : LOW;
#endif
      }
      return LOW;
    }

    static void callback(PORT port, callback_t func)
    {
      _callback[port] = func;
    }

    static void interrupt(PIN pin, ISC mode)
    {
      pindef_t spin = { .port = pin };
      PORT_t* port = getPort(spin.x.port);
      if (port)
      {
        register8_t &ctrl = ((register8_t *)&port->PIN0CTRL)[spin.x.bpos];
        if (mode != ISC_INTDISABLE)
        {
#if CONFIG_PORT_ISR
          ctrl = (ctrl & ~PORT_ISC_gm) | mode;
#endif
        }
        else
        {
          ctrl = (ctrl & ~PORT_ISC_gm) | ISC_INTDISABLE;
        }
      }
    }

    static void poll(void)
    {
#if !CONFIG_TCA_ISR
#if defined(PORTA)
      isr_port(0, VPORTA);
#endif
#if defined(PORTB)
      isr_port(1, VPORTB);
#endif
#if defined(PORTC)
      isr_port(2, VPORTC);
#endif
#if defined(PORTD)
      isr_port(3, VPORTD);
#endif
#if defined(PORTE)
      isr_port(4, VPORTE);
#endif
#if defined(PORTF)
      isr_port(5, VPORTF);
#endif
#if defined(PORTG)
      isr_port(6, VPORTG);
#endif
#endif
    }

  protected:

    typedef union
    {
      struct
      {
        uint8_t port: 4;
        uint8_t bpos: 3;
        uint8_t ____: 1;
        uint8_t bmsk: 8;
      } x;
      uint16_t port;
    } pindef_t;

    static callback_t _callback[PORT_MAX_PORTS];

#if CONFIG_PORT_ISR
    friend void port_isr_port(uint8_t ch, VPORT_t &port);
#endif

    static void isr_port(uint8_t ch, VPORT_t &port)
    {
      uint8_t flags = port.INTFLAGS = port.INTFLAGS;
      if (flags && _callback[ch])
        _callback[ch](port.IN, flags);
    }

    static PORT_t *getPort(uint8_t port)
    {
      if (port < PORT_MAX_PORTS)
      {
#if (1 < PORT_MAX_POTS) && !defined(PORTB)
        if (port != 1)
#endif
        return &PORTA + port;
      }
      return 0;
    }
};
