/*
  avr8_portmux.h - PORTMUX Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include <stddef.h>
#include "avr8_config.h"
#include "avr8_portmux_dfp.h"

#define PORTMUX_REG(name) ((uint8_t)offsetof(PORTMUX_t, name) << 4)

class PortMux /* AVR0 Series */
{
  public:

#if defined(__AVR_0__)

    /* AVR0 Series */

    typedef enum
    {
      ALT_EVOUTA = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT0_bp,
      ALT_EVOUTB = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT1_bp,
      ALT_EVOUTC = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT2_bp,
      ALT_LUT0   = PORTMUX_REG(CTRLA) + PORTMUX_LUT0_bp,
      ALT_LUT1   = PORTMUX_REG(CTRLA) + PORTMUX_LUT1_bp,
      ALT_USART0 = PORTMUX_REG(CTRLB) + PORTMUX_USART0_bp,
      ALT_SPI0   = PORTMUX_REG(CTRLB) + PORTMUX_SPI0_bp,
      ALT_TCA0_0 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_2_bp,
      ALT_TCA0_3 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_3_bp,
  #if defined(PORTMUX_TCA04_bp)
      ALT_TCA0_4 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_4_bp,
  #endif
  #if defined(PORTMUX_TCA05_bp)
      ALT_TCA0_5 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_5_bp,
  #endif
      ALT_TCB0   = PORTMUX_REG(CTRLD) + PORTMUX_TCB0_bp,
    } ALT;

    static void reset(void)
    {
      PORTMUX.CTRLA = 0;
      PORTMUX.CTRLB = 0;
      PORTMUX.CTRLC = 0;
      PORTMUX.CTRLD = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_1__)

    /* AVR1 Series */

    typedef enum
    {
      ALT_EVOUTA = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT0_bp,
      ALT_EVOUTB = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT1_bp,
      ALT_EVOUTC = PORTMUX_REG(CTRLA) + PORTMUX_EVOUT2_bp,
      ALT_LUT0   = PORTMUX_REG(CTRLA) + PORTMUX_LUT0_bp,
      ALT_LUT1   = PORTMUX_REG(CTRLA) + PORTMUX_LUT1_bp,
      ALT_USART0 = PORTMUX_REG(CTRLB) + PORTMUX_USART0_bp,
      ALT_SPI0   = PORTMUX_REG(CTRLB) + PORTMUX_SPI0_bp,
  #if defined(PORTMUX_TWI0_bp)
      ALT_TWI0   = PORTMUX_REG(CTRLB) + PORTMUX_TWI0_bp,
  #endif
      ALT_TCA0_0 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_2_bp,
      ALT_TCA0_3 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_3_bp,
  #if defined(PORTMUX_TCA04_bp)
      ALT_TCA0_4 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_4_bp,
  #endif
  #if defined(PORTMUX_TCA05_bp)
      ALT_TCA0_5 = PORTMUX_REG(CTRLC) + PORTMUX_TCA0_5_bp,
  #endif
      ALT_TCB0   = PORTMUX_REG(CTRLD) + PORTMUX_TCB0_bp,
  #if defined(PORTMUX_TCB1_bp)
      ALT_TCB1   = PORTMUX_REG(CTRLD) + PORTMUX_TCB1_bp,
  #endif
    } ALT;

    static void reset(void)
    {
      PORTMUX.CTRLA = 0;
      PORTMUX.CTRLB = 0;
      PORTMUX.CTRLC = 0;
      PORTMUX.CTRLD = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX.CTRLA)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_2__)

    /* AVR2 Series */

    typedef enum
    {
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
      ALT_EVOUTB   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTB_bp,
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTC_bp,
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
      ALT_TCA0_3   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_3_bp,
      ALT_TCA0_4   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_4_bp,
      ALT_TCA0_5   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_5_bp,
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA  = 0;
      PORTMUX.CCLROUTEA    = 0;
      PORTMUX.USARTROUTEA  = 0;
      PORTMUX.SPIROUTEA    = 0;
      PORTMUX.TCAROUTEA    = 0;
      PORTMUX.TCBROUTEA    = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_DA__)

    /* AVR-DA Series */

    typedef enum
    {
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #if defined(PORTMUX_EVOUTB_bp)
      ALT_EVOUTB   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTB_bp,
  #endif
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTC_bp,
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
  #if defined(PORTMUX_EVOUTE_bp)
      ALT_EVOUTE   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTE_bp,
  #endif
  #if defined(PORTMUX_EVOUTF_bp)
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
  #endif
  #if defined(PORTMUX_EVOUTG_bp)
      ALT_EVOUTG   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTG_bp,
  #endif
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
  #if defined(PORTMUX_LUT4_bp)
      ALT_LUT4     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT4_bp,
  #endif
  #if defined(PORTMUX_LUT5_bp)
      ALT_LUT5     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT5_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_1_bp,
      ALT_USART2_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART2_0_bp,
      ALT_USART2_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART2_1_bp,
  #if defined(PORTMUX_USART3_bp)
      ALT_USART3_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART3_0_bp,
      ALT_USART3_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART3_1_bp,
  #endif
  #if defined(PORTMUX_USART4_bp)
      ALT_USART4_0 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART4_0_bp,
      ALT_USART4_1 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART4_1_bp,
  #endif
  #if defined(PORTMUX_USART5_bp)
      ALT_USART5_0 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART5_0_bp,
      ALT_USART5_1 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART5_1_bp,
  #endif
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI1_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI1_0_bp,
      ALT_SPI1_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI1_1_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
  #if defined(PORTMUX_TWI1_bp)
      ALT_TWI1_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI1_0_bp,
      ALT_TWI1_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI1_1_bp,
  #endif
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
  #if defined(PORTMUX_TCA1_bp)
      ALT_TCA1_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_0_bp,
      ALT_TCA1_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_1_bp,
      ALT_TCA1_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_2_bp,
  #endif
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
      ALT_TCB2     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB2_bp,
  #if defined(PORTMUX_TCB3_bp)
      ALT_TCB3     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB3_bp,
  #endif
  #if defined(PORTMUX_TCB4_bp)
      ALT_TCB4     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB4_bp,
  #endif
      ALT_TCD0_0   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_0_bp,
      ALT_TCD0_1   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_1_bp,
      ALT_TCD0_2   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_2_bp,
      ALT_AC0      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC0_bp,
      ALT_AC1      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC1_bp,
      ALT_AC2      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC2_bp,
      ALT_ZCD0     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD0_bp,
  #if defined(PORTMUX_ZCD1_bp)
      ALT_ZCD1     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD1_bp,
  #endif
  #if defined(PORTMUX_ZCD2_bp)
      ALT_ZCD2     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD2_bp,
  #endif
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA  = 0;
      PORTMUX.CCLROUTEA    = 0;
      PORTMUX.USARTROUTEA  = 0;
  #if defined(PORTMUX_USARTROUTEB)
      PORTMUX.USARTROUTEB  = 0;
  #endif
      PORTMUX.SPIROUTEA    = 0;
      PORTMUX.TWIROUTEA    = 0;
      PORTMUX.TCAROUTEA    = 0;
      PORTMUX.TCBROUTEA    = 0;
      PORTMUX.TCDROUTEA    = 0;
      PORTMUX.ACROUTEA     = 0;
      PORTMUX.ZCDROUTEA    = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_DB__)

    /* AVR-DB Series */

    typedef enum
    {
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #if defined(PORTMUX_EVOUTB_bp)
      ALT_EVOUTB   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTB_bp,
  #endif
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTC_bp,
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
  #if defined(PORTMUX_EVOUTE_bp)
      ALT_EVOUTE   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTE_bp,
  #endif
  #if defined(PORTMUX_EVOUTF_bp)
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
  #endif
  #if defined(PORTMUX_EVOUTG_bp)
      ALT_EVOUTG   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTG_bp,
  #endif
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
  #if defined(PORTMUX_LUT3_bp)
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
  #endif
  #if defined(PORTMUX_LUT4_bp)
      ALT_LUT4     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT4_bp,
  #endif
  #if defined(PORTMUX_LUT5_bp)
      ALT_LUT5     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT5_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_1_bp,
      ALT_USART2_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART2_0_bp,
      ALT_USART2_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART2_1_bp,
  #if defined(PORTMUX_USART3_bp)
      ALT_USART3_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART3_0_bp,
      ALT_USART3_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART3_1_bp,
  #endif
  #if defined(PORTMUX_USART4_bp)
      ALT_USART4_0 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART4_0_bp,
      ALT_USART4_1 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART4_1_bp,
  #endif
  #if defined(PORTMUX_USART5_bp)
      ALT_USART5_0 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART5_0_bp,
      ALT_USART5_1 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART5_1_bp,
  #endif
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI1_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI1_0_bp,
      ALT_SPI1_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI1_1_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
  #if defined(PORTMUX_TWI1_bp)
      ALT_TWI1_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI1_0_bp,
      ALT_TWI1_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI1_1_bp,
  #endif
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
  #if defined(PORTMUX_TCA1_bp)
      ALT_TCA1_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_0_bp,
      ALT_TCA1_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_1_bp,
      ALT_TCA1_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_2_bp,
  #endif
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
      ALT_TCB2     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB2_bp,
  #if defined(PORTMUX_TCB3_bp)
      ALT_TCB3     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB3_bp,
  #endif
  #if defined(PORTMUX_TCB4_bp)
      ALT_TCB4     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB4_bp,
  #endif
      ALT_TCD0_0   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_0_bp,
      ALT_TCD0_1   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_1_bp,
      ALT_TCD0_2   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_2_bp,
      ALT_AC0      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC0_bp,
      ALT_AC1      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC1_bp,
      ALT_AC2      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC2_bp,
      ALT_ZCD0     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD0_bp,
  #if defined(PORTMUX_ZCD1_bp)
      ALT_ZCD1     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD1_bp,
  #endif
  #if defined(PORTMUX_ZCD2_bp)
      ALT_ZCD2     = PORTMUX_REG(ZCDROUTEA)   + PORTMUX_ZCD2_bp,
  #endif
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA  = 0;
      PORTMUX.CCLROUTEA    = 0;
      PORTMUX.USARTROUTEA  = 0;
  #if defined(PORTMUX_USARTROUTEB)
      PORTMUX.USARTROUTEB  = 0;
  #endif
      PORTMUX.SPIROUTEA    = 0;
      PORTMUX.TWIROUTEA    = 0;
      PORTMUX.TCAROUTEA    = 0;
      PORTMUX.TCBROUTEA    = 0;
      PORTMUX.TCDROUTEA    = 0;
      PORTMUX.ACROUTEA     = 0;
      PORTMUX.ZCDROUTEA    = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_DD__)

    /* AVR-DD Series */

    typedef enum
    {
  #if defined(PORTMUX_EVOUTA_bp)
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #endif
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
  #if defined(PORTMUX_EVOUTF_bp)
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
  #endif
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
  #if defined(PORTMUX_LUT3_bp)
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART0_2 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_2_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_1_bp,
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI0_2   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_2_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
  #if defined(PORTMUX_TCB0_bp)
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
  #endif
  #if defined(PORTMUX_TCB1_bp)
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
  #endif
      ALT_TCD0_0   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_0_bp,
      ALT_TCD0_1   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_1_bp,
      ALT_TCD0_2   = PORTMUX_REG(TCDROUTEA)   + PORTMUX_TCD0_2_bp,
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA  = 0;
      PORTMUX.CCLROUTEA    = 0;
      PORTMUX.USARTROUTEA  = 0;
      PORTMUX.SPIROUTEA    = 0;
      PORTMUX.TWIROUTEA    = 0;
      PORTMUX.TCAROUTEA    = 0;
  #if defined(PORTMUX_TCBROUTEA)
      PORTMUX.TCBROUTEA    = 0;
  #endif
      PORTMUX.TCDROUTEA    = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_DU__)

    /* AVR-DU Series */

    typedef enum
    {
  #if defined(PORTMUX_EVOUTA_bp)
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #endif
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
  #if defined(PORTMUX_LUT3_bp)
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART0_2 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_2_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_1_bp,
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI0_2   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_2_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
  #if defined(PORTMUX_TCB0_bp)
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
  #endif
  #if defined(PORTMUX_TCB1_bp)
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
  #endif
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA  = 0;
      PORTMUX.CCLROUTEA    = 0;
      PORTMUX.USARTROUTEA  = 0;
      PORTMUX.SPIROUTEA    = 0;
      PORTMUX.TWIROUTEA    = 0;
      PORTMUX.TCAROUTEA    = 0;
  #if defined(PORTMUX_TCBROUTEA)
      PORTMUX.TCBROUTEA    = 0;
  #endif
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_EA__)

    /* AVR-EA Series */

    typedef enum
    {
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #if defined(PORTMUX_EVOUTB_bp)
      ALT_EVOUTB   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTB_bp,
  #endif
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTC_bp,
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
  #if defined(PORTMUX_EVOUTE_bp)
      ALT_EVOUTE   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTE_bp,
  #endif
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART0_2 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_2_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART1_1_bp,
      ALT_USART2_0 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART2_0_bp,
      ALT_USART2_1 = PORTMUX_REG(USARTROUTEB) + PORTMUX_USART2_1_bp,
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI0_2   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_2_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA0_2_bp,
      ALT_TCA1_0   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_0_bp,
      ALT_TCA1_1   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_1_bp,
      ALT_TCA1_2   = PORTMUX_REG(TCAROUTEA)   + PORTMUX_TCA1_2_bp,
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
      ALT_TCB2     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB2_bp,
      ALT_TCB3     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB3_bp,
      ALT_AC0      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC0_bp,
      ALT_AC1      = PORTMUX_REG(ACROUTEA)    + PORTMUX_AC1_bp
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA   = 0;
      PORTMUX.CCLROUTEA     = 0;
      PORTMUX.USARTROUTEA   = 0;
      PORTMUX.USARTROUTEB   = 0;
      PORTMUX.SPIROUTEA     = 0;
      PORTMUX.TWIROUTEA     = 0;
      PORTMUX.TCAROUTEA     = 0;
      PORTMUX.TCBROUTEA     = 0;
      PORTMUX.ACROUTEA      = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_EB__)

    /* AVR-EB Series */

    typedef enum
    {
  #if defined(PORTMUX_EVOUTA_bp)
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTA_bp,
  #endif
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTC_bp,
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTD_bp,
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA) + PORTMUX_EVOUTF_bp,
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT2_bp,
  #if defined(PORTMUX_LUT3_bp)
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)   + PORTMUX_LUT3_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_1_bp,
      ALT_USART0_2 = PORTMUX_REG(USARTROUTEA) + PORTMUX_USART0_2_bp,
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_1_bp,
      ALT_SPI0_2   = PORTMUX_REG(SPIROUTEA)   + PORTMUX_SPI0_2_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)   + PORTMUX_TWI0_1_bp,
      ALT_TCE0_0   = PORTMUX_REG(TCEROUTEA)   + PORTMUX_TCE0_0_bp,
      ALT_TCE0_1   = PORTMUX_REG(TCEROUTEA)   + PORTMUX_TCE0_1_bp,
      ALT_TCE0_2   = PORTMUX_REG(TCEROUTEA)   + PORTMUX_TCE0_2_bp,
      ALT_TCE0_3   = PORTMUX_REG(TCEROUTEA)   + PORTMUX_TCE0_3_bp,
  #if defined(PORTMUX_TCB0_bp)
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB0_bp,
  #endif
  #if defined(PORTMUX_TCB1_bp)
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)   + PORTMUX_TCB1_bp,
  #endif
      ALT_TCF0_0   = PORTMUX_REG(TCFROUTEA)   + PORTMUX_TCF0_0_bp,
      ALT_TCF0_1   = PORTMUX_REG(TCFROUTEA)   + PORTMUX_TCF0_1_bp,
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA   = 0;
      PORTMUX.CCLROUTEA     = 0;
      PORTMUX.USARTROUTEA   = 0;
      PORTMUX.SPIROUTEA     = 0;
      PORTMUX.TWIROUTEA     = 0;
      PORTMUX.TCEROUTEA     = 0;
  #if defined(PORTMUX_TCBROUTEA)
      PORTMUX.TCBROUTEA     = 0;
  #endif
      PORTMUX.TCFROUTEA     = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#elif defined(__AVR_SD__)

    /* AVR-Sx Series */

    typedef enum
    {
      ALT_EVOUTA   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTA_bp,
  #if defined(PORTMUX_EVOUTB_bp)
      ALT_EVOUTB   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTB_bp,
  #endif
      ALT_EVOUTC   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTC_bp,
      ALT_EVOUTD   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTD_bp,
  #if defined(PORTMUX_EVOUTE_bp)
      ALT_EVOUTE   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTE_bp,
  #endif
  #if defined(PORTMUX_EVOUTF_bp)
      ALT_EVOUTF   = PORTMUX_REG(EVSYSROUTEA)   + PORTMUX_EVOUTF_bp,
  #endif
      ALT_LUT0     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT0_bp,
      ALT_LUT1     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT1_bp,
      ALT_LUT2     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT2_bp,
  #if defined(PORTMUX_LUT3_bp)
      ALT_LUT3     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT3_bp,
  #endif
  #if defined(PORTMUX_LUT4_bp)
      ALT_LUT4     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT4_bp,
  #endif
  #if defined(PORTMUX_LUT5_bp)
      ALT_LUT5     = PORTMUX_REG(CCLROUTEA)     + PORTMUX_LUT5_bp,
  #endif
      ALT_USART0_0 = PORTMUX_REG(USARTROUTEA)   + PORTMUX_USART0_0_bp,
      ALT_USART0_1 = PORTMUX_REG(USARTROUTEA)   + PORTMUX_USART0_1_bp,
      ALT_USART0_2 = PORTMUX_REG(USARTROUTEA)   + PORTMUX_USART0_2_bp,
      ALT_USART1_0 = PORTMUX_REG(USARTROUTEA)   + PORTMUX_USART1_0_bp,
      ALT_USART1_1 = PORTMUX_REG(USARTROUTEA)   + PORTMUX_USART1_1_bp,
  #if defined(PORTMUX_USART2_0_bp)
      ALT_USART2_0 = PORTMUX_REG(USARTROUTEB)   + PORTMUX_USART2_0_bp,
  #endif
  #if defined(PORTMUX_USART2_1_bp)
      ALT_USART2_1 = PORTMUX_REG(USARTROUTEB)   + PORTMUX_USART2_1_bp,
  #endif
      ALT_SPI0_0   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI0_0_bp,
      ALT_SPI0_1   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI0_1_bp,
      ALT_SPI0_2   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI0_2_bp,
      ALT_SPI1_0   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI1_0_bp,
      ALT_SPI1_1   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI1_1_bp,
      ALT_SPI1_2   = PORTMUX_REG(SPIROUTEA)     + PORTMUX_SPI1_2_bp,
      ALT_TWI0_0   = PORTMUX_REG(TWIROUTEA)     + PORTMUX_TWI0_0_bp,
      ALT_TWI0_1   = PORTMUX_REG(TWIROUTEA)     + PORTMUX_TWI0_1_bp,
  #if defined(PORTMUX_TWI1_bp)
      ALT_TWI1_0   = PORTMUX_REG(TWIROUTEA)     + PORTMUX_TWI1_0_bp,
      ALT_TWI1_1   = PORTMUX_REG(TWIROUTEA)     + PORTMUX_TWI1_1_bp,
  #endif
      ALT_TCA0_0   = PORTMUX_REG(TCAROUTEA)     + PORTMUX_TCA0_0_bp,
      ALT_TCA0_1   = PORTMUX_REG(TCAROUTEA)     + PORTMUX_TCA0_1_bp,
      ALT_TCA0_2   = PORTMUX_REG(TCAROUTEA)     + PORTMUX_TCA0_2_bp,
      ALT_TCB0     = PORTMUX_REG(TCBROUTEA)     + PORTMUX_TCB0_bp,
      ALT_TCB1     = PORTMUX_REG(TCBROUTEA)     + PORTMUX_TCB1_bp,
  #if defined(PORTMUX_TCB2_bp)
      ALT_TCB2     = PORTMUX_REG(TCBROUTEA)     + PORTMUX_TCB2_bp,
  #endif
      ALT_TCB3     = PORTMUX_REG(TCBROUTEA)     + PORTMUX_TCB3_bp,
      ALT_TCD0_0   = PORTMUX_REG(TCDROUTEA)     + PORTMUX_TCD0_0_bp,
      ALT_TCD0_1   = PORTMUX_REG(TCDROUTEA)     + PORTMUX_TCD0_1_bp,
      ALT_TCD0_2   = PORTMUX_REG(TCDROUTEA)     + PORTMUX_TCD0_2_bp,
      ALT_AC0      = PORTMUX_REG(ACROUTEA)      + PORTMUX_AC0_bp,
      ALT_AC1      = PORTMUX_REG(ACROUTEA)      + PORTMUX_AC1_bp,
      ALT_AC2      = PORTMUX_REG(ACROUTEA)      + PORTMUX_AC2_bp,
  #if defined(PORTMUX_ZCD0_bp)
      ALT_ZCD0     = PORTMUX_REG(ZCDROUTEA)     + PORTMUX_ZCD0_bp,
  #endif
      ALT_ZCD3     = PORTMUX_REG(ZCDROUTEA)     + PORTMUX_ZCD3_bp,
      ALT_HEART_0  = PORTMUX_REG(ERRCTRLROUTEA) + PORTMUX_HEART_0_bp,
      ALT_HEART_1  = PORTMUX_REG(ERRCTRLROUTEA) + PORTMUX_HEART_1_bp,
      ALT_HEART_2  = PORTMUX_REG(ERRCTRLROUTEA) + PORTMUX_HEART_2_bp,
    } ALT;

    static void reset(void)
    {
      PORTMUX.EVSYSROUTEA   = 0;
      PORTMUX.CCLROUTEA     = 0;
      PORTMUX.USARTROUTEA   = 0;
  #if defined(PORTMUX_USARTROUTEB)
      PORTMUX.USARTROUTEB   = 0;
  #endif
      PORTMUX.SPIROUTEA     = 0;
      PORTMUX.TWIROUTEA     = 0;
      PORTMUX.TCAROUTEA     = 0;
      PORTMUX.TCBROUTEA     = 0;
      PORTMUX.TCDROUTEA     = 0;
      PORTMUX.ACROUTEA      = 0;
      PORTMUX.ZCDROUTEA     = 0;
      PORTMUX.ERRCTRLROUTEA = 0;
    }

    static void alternate(ALT alt, bool enable = true)
    {
      uint8_t bit = _BV(alt & 7);
      register8_t &reg = ((register8_t *)&PORTMUX)[(uint8_t)alt >> 4];
      if (enable)
        reg |= bit;
      else
        reg &= ~bit;
    }

#endif
};
