/*
  avr8_rtc.h - RTC Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_rtc_dfp.h"

#if CONFIG_RTC_USE || ((CONFIG_TASK_USE || CONFIG_ALARM_USE) && (CONFIG_ALARM_TIMER == TIMER_RTC))

class Rtc
{
  public:

    /* RTC Clock Select select */
    typedef enum RTC_CLKSEL_enum
    {
#if defined(CLKCTRL_OSCHFCTRLA)
      CLKSEL_OSC32K  = RTC_CLKSEL_OSC32K_gc,   /* 32.768 kHz from OSC32K */
      CLKSEL_OSC1K   = RTC_CLKSEL_OSC1K_gc,    /* 1.024 kHz from OSC32K */
      CLKSEL_XOSC32K = RTC_CLKSEL_XOSC32K_gc,  /* 32.768 kHz from XOSC32K */
      CLKSEL_EXTCLK  = RTC_CLKSEL_EXTCLK_gc,   /* External Clock */
#else
      CLKSEL_OSC32K = RTC_CLKSEL_INT32K_gc,  /* Internal 32kHz OSC */
      CLKSEL_OSC1K  = RTC_CLKSEL_INT1K_gc,   /* Internal 1kHz OSC */
      CLKSEL_EXTCLK = RTC_CLKSEL_EXTCLK_gc,  /* External Clock */
#endif
    } CLKSEL;

    /* RTC Prescaling Factor select */
    typedef enum
    {
      PRESCALER_DIV1     = RTC_PRESCALER_DIV1_gc,      /* RTC Clock / 1 */
      PRESCALER_DIV2     = RTC_PRESCALER_DIV2_gc,      /* RTC Clock / 2 */
      PRESCALER_DIV4     = RTC_PRESCALER_DIV4_gc,      /* RTC Clock / 4 */
      PRESCALER_DIV8     = RTC_PRESCALER_DIV8_gc,      /* RTC Clock / 8 */
      PRESCALER_DIV16    = RTC_PRESCALER_DIV16_gc,     /* RTC Clock / 16 */
      PRESCALER_DIV32    = RTC_PRESCALER_DIV32_gc,     /* RTC Clock / 32 */
      PRESCALER_DIV64    = RTC_PRESCALER_DIV64_gc,     /* RTC Clock / 64 */
      PRESCALER_DIV128   = RTC_PRESCALER_DIV128_gc,    /* RTC Clock / 128 */
      PRESCALER_DIV256   = RTC_PRESCALER_DIV256_gc,    /* RTC Clock / 256 */
      PRESCALER_DIV512   = RTC_PRESCALER_DIV512_gc,    /* RTC Clock / 512 */
      PRESCALER_DIV1024  = RTC_PRESCALER_DIV1024_gc,   /* RTC Clock / 1024 */
      PRESCALER_DIV2048  = RTC_PRESCALER_DIV2048_gc,   /* RTC Clock / 2048 */
      PRESCALER_DIV4096  = RTC_PRESCALER_DIV4096_gc,   /* RTC Clock / 4096 */
      PRESCALER_DIV8192  = RTC_PRESCALER_DIV8192_gc,   /* RTC Clock / 8192 */
      PRESCALER_DIV16384 = RTC_PRESCALER_DIV16384_gc,  /* RTC Clock / 16384 */
      PRESCALER_DIV32768 = RTC_PRESCALER_DIV32768_gc,  /* RTC Clock / 32768 */
    } PRESCALER;

    /* PIT Period select */
    typedef enum
    {
      PERIOD_OFF      = RTC_PERIOD_OFF_gc,       /* Off */
      PERIOD_CYC4     = RTC_PERIOD_CYC4_gc,      /* RTC Clock Cycles 4 */
      PERIOD_CYC8     = RTC_PERIOD_CYC8_gc,      /* RTC Clock Cycles 8 */
      PERIOD_CYC16    = RTC_PERIOD_CYC16_gc,     /* RTC Clock Cycles 16 */
      PERIOD_CYC32    = RTC_PERIOD_CYC32_gc,     /* RTC Clock Cycles 32 */
      PERIOD_CYC64    = RTC_PERIOD_CYC64_gc,     /* RTC Clock Cycles 64 */
      PERIOD_CYC128   = RTC_PERIOD_CYC128_gc,    /* RTC Clock Cycles 128 */
      PERIOD_CYC256   = RTC_PERIOD_CYC256_gc,    /* RTC Clock Cycles 256 */
      PERIOD_CYC512   = RTC_PERIOD_CYC512_gc,    /* RTC Clock Cycles 512 */
      PERIOD_CYC1024  = RTC_PERIOD_CYC1024_gc,   /* RTC Clock Cycles 1024 */
      PERIOD_CYC2048  = RTC_PERIOD_CYC2048_gc,   /* RTC Clock Cycles 2048 */
      PERIOD_CYC4096  = RTC_PERIOD_CYC4096_gc,   /* RTC Clock Cycles 4096 */
      PERIOD_CYC8192  = RTC_PERIOD_CYC8192_gc,   /* RTC Clock Cycles 8192 */
      PERIOD_CYC16384 = RTC_PERIOD_CYC16384_gc,  /* RTC Clock Cycles 16384 */
      PERIOD_CYC32768 = RTC_PERIOD_CYC32768_gc,  /* RTC Clock Cycles 32768 */
    } PERIOD;

#if defined(RTC_EVGEN0SEL_gm)
    /* Event Generation 0 Select */
    typedef enum
    {
      EVGEN0SEL_OFF      = RTC_EVGEN0SEL_OFF_gc,       /* No Event Generated */
      EVGEN0SEL_DIV4     = RTC_EVGEN0SEL_DIV4_gc,      /* CLK_RTC divided by 4 */
      EVGEN0SEL_DIV8     = RTC_EVGEN0SEL_DIV8_gc,      /* CLK_RTC divided by 8 */
      EVGEN0SEL_DIV16    = RTC_EVGEN0SEL_DIV16_gc,     /* CLK_RTC divided by 16 */
      EVGEN0SEL_DIV32    = RTC_EVGEN0SEL_DIV32_gc,     /* CLK_RTC divided by 32 */
      EVGEN0SEL_DIV64    = RTC_EVGEN0SEL_DIV64_gc,     /* CLK_RTC divided by 64 */
      EVGEN0SEL_DIV128   = RTC_EVGEN0SEL_DIV128_gc,    /* CLK_RTC divided by 128 */
      EVGEN0SEL_DIV256   = RTC_EVGEN0SEL_DIV256_gc,    /* CLK_RTC divided by 256 */
      EVGEN0SEL_DIV512   = RTC_EVGEN0SEL_DIV512_gc,    /* CLK_RTC divided by 512 */
      EVGEN0SEL_DIV1024  = RTC_EVGEN0SEL_DIV1024_gc,   /* CLK_RTC divided by 1024 */
      EVGEN0SEL_DIV2048  = RTC_EVGEN0SEL_DIV2048_gc,   /* CLK_RTC divided by 2048 */
      EVGEN0SEL_DIV4096  = RTC_EVGEN0SEL_DIV4096_gc,   /* CLK_RTC divided by 4096 */
      EVGEN0SEL_DIV8192  = RTC_EVGEN0SEL_DIV8192_gc,   /* CLK_RTC divided by 8192 */
      EVGEN0SEL_DIV16384 = RTC_EVGEN0SEL_DIV16384_gc,  /* CLK_RTC divided by 16384 */
      EVGEN0SEL_DIV32768 = RTC_EVGEN0SEL_DIV32768_gc,  /* CLK_RTC divided by 32768 */
    } EVGEN0SEL;

    /* Event Generation 1 Select */
    typedef enum
    {
      EVGEN1SEL_OFF      = RTC_EVGEN1SEL_OFF_gc,       /* No Event Generated */
      EVGEN1SEL_DIV4     = RTC_EVGEN1SEL_DIV4_gc,      /* CLK_RTC divided by 4 */
      EVGEN1SEL_DIV8     = RTC_EVGEN1SEL_DIV8_gc,      /* CLK_RTC divided by 8 */
      EVGEN1SEL_DIV16    = RTC_EVGEN1SEL_DIV16_gc,     /* CLK_RTC divided by 16 */
      EVGEN1SEL_DIV32    = RTC_EVGEN1SEL_DIV32_gc,     /* CLK_RTC divided by 32 */
      EVGEN1SEL_DIV64    = RTC_EVGEN1SEL_DIV64_gc,     /* CLK_RTC divided by 64 */
      EVGEN1SEL_DIV128   = RTC_EVGEN1SEL_DIV128_gc,    /* CLK_RTC divided by 128 */
      EVGEN1SEL_DIV256   = RTC_EVGEN1SEL_DIV256_gc,    /* CLK_RTC divided by 256 */
      EVGEN1SEL_DIV512   = RTC_EVGEN1SEL_DIV512_gc,    /* CLK_RTC divided by 512 */
      EVGEN1SEL_DIV1024  = RTC_EVGEN1SEL_DIV1024_gc,   /* CLK_RTC divided by 1024 */
      EVGEN1SEL_DIV2048  = RTC_EVGEN1SEL_DIV2048_gc,   /* CLK_RTC divided by 2048 */
      EVGEN1SEL_DIV4096  = RTC_EVGEN1SEL_DIV4096_gc,   /* CLK_RTC divided by 4096 */
      EVGEN1SEL_DIV8192  = RTC_EVGEN1SEL_DIV8192_gc,   /* CLK_RTC divided by 8192 */
      EVGEN1SEL_DIV16384 = RTC_EVGEN1SEL_DIV16384_gc,  /* CLK_RTC divided by 16384 */
      EVGEN1SEL_DIV32768 = RTC_EVGEN1SEL_DIV32768_gc,  /* CLK_RTC divided by 32768 */
    } EVGEN1SEL;
#endif

    /* Type of Callback Function */
    typedef void (*callback_t)(void);

    static void begin(CLKSEL clksel = CLKSEL_OSC32K, PRESCALER prescaler = PRESCALER_DIV1)
    {
      end();
#if defined(RTC_EVGEN0SEL_gm)
      RTC.PITEVGENCTRLA = 0;
#endif
#if defined(RTC_ERROR_gm)
      RTC.CALIB         = 0;
#endif
      RTC.PITDBGCTRL    = 0;
      RTC.PITINTFLAGS   = RTC_PI_bm;
      RTC.PITINTCTRL    = 0;
      RTC.DBGCTRL       = 0;
      RTC.INTFLAGS      = RTC_OVF_bm | RTC_CMP_bm;
      RTC.INTCTRL       = RTC_OVF_bm;
      RTC.CLKSEL        = clksel;
      regCNT(0);
      regPER(0xFFFF);
      regCTRLA(prescaler);
    }

    static void end(void)
    {
      regCTRLA(0);
      regPITCTRLA(0);
    }

    static void runstdby(bool enable = true)
    {
      regCTRLA((RTC.CTRLA & ~RTC_RUNSTDBY_bm) | (enable ? RTC_RUNSTDBY_bm : 0));
    }

    static void dbgctrl(bool dbgrun = false)
    {
      RTC.PITDBGCTRL = dbgrun ? RTC_DBGRUN_bm : 0;
      RTC.DBGCTRL = dbgrun ? RTC_DBGRUN_bm : 0;
    }

    static void run(bool enable = true)
    {
      regCTRLA((RTC.CTRLA & ~RTC_RTCEN_bm) | (enable ? RTC_RTCEN_bm : 0));
    }

#if defined(RTC_ERROR_gm)
    static void calib(int8_t ppm = 0)
    {
      RTC.CALIB = ppm;
      regCTRLA((RTC.CTRLA & ~RTC_CORREN_bm) | (ppm ? RTC_CORREN_bm : 0));
    }
#endif

#if defined(RTC_EVGEN0SEL_gm)
    static void event(EVGEN0SEL evgen0 = EVGEN0SEL_OFF, EVGEN1SEL evgen1 = EVGEN1SEL_OFF)
    {
      RTC.PITEVGENCTRLA = evgen1 | evgen0;
    }
#endif

    static void callback(callback_t func)
    {
      _callback[0] = func;
    }

    static void interval(uint32_t tick, bool fromnow = false)
    {
      _interval = tick <= 5 ? 5 : tick;
      if (fromnow)
        regCMP(RTC.CNT);
    }

    static void interrupt(uint32_t tick, bool restart = false)
    {
      if (tick)
      {
        interval(tick, true);
        if (restart || ((RTC.INTCTRL & RTC_CMP_bm) == 0))
        {
          setup();
          RTC.INTCTRL |= (RTC.INTFLAGS = RTC_CMP_bm);
        }
      }        
      else
      {
        RTC.INTCTRL &= ~RTC_CMP_bm;
      }
    }

    static void callbackPIT(callback_t func)
    {
      _callback[1] = func;
    }

    static void intervalPIT(PERIOD period)
    {
      regPITCTRLA((RTC.PITCTRLA & ~(RTC_PERIOD_gm | RTC_PITEN_bm)) | period | (period != PERIOD_OFF ? RTC_PITEN_bm : 0));
    }

    static void interruptPIT(PERIOD period)
    {
      intervalPIT(period);
      if (period != PERIOD_OFF)
        RTC.PITINTCTRL = (RTC.PITINTFLAGS = RTC_PI_bm);
      else
        RTC.PITINTCTRL = 0;
    }

    static uint32_t read(void)
    {
      union
      {
        uint32_t d;
        struct { uint16_t l, h; } w;
      } x;
      do
      {
        x.w.h = _overflow;
        x.w.l = RTC.CNT;
      }
      while (x.w.h != _overflow);
      return x.d;
    }

    static void delay(uint32_t tick)
    {
      for (uint32_t t = read(); read() - t < tick; )
        yield();
    }

    static uint32_t us2tick(uint16_t us)
    {
      return (uint32_t)(us) * 512 / 15625;
    }

    static uint32_t us2tick(uint32_t us)
    {
      return (uint64_t)(us) * 512 / 15625;
    }

    static uint32_t tick2us(uint16_t tick)
    {
      return (uint32_t)(tick) * 15625 / 512;
    }

    static uint32_t tick2us(uint32_t tick)
    {
      return (uint64_t)(tick) * 15625 / 512;
    }

    static void poll(void)
    {
    }

  protected:

    static uint16_t          _khz;
    static volatile uint16_t _overflow;
    static volatile uint16_t _counter;
    static uint32_t          _interval;
    static callback_t        _callback[2];

    friend void rtc_isr_cnt(void);
    friend void rtc_isr_pit(void);

    inline static void isr_cnt(void)
    {
      uint8_t flags = RTC.INTFLAGS = RTC.INTFLAGS;
      if (flags & RTC_OVF_bm)
        ++_overflow;
      if (flags & RTC_CMP_bm)
      {
        if (--_counter)
          regCMP(RTC.CMP + 0x8000);
        else
        {
          if (_callback[0])
            _callback[0]();
          setup();
        }
      }
    }

    inline static void isr_pit(void)
    {
      if ((RTC.PITINTFLAGS = RTC.PITINTFLAGS) & RTC_PI_bm)
      {
        if (_callback[1])
          _callback[1]();
      }          
    }

    static void setup(void) __attribute__((noinline))
    {
      int16_t next = (int16_t)_interval;
      if ((_counter = ((uint16_t)(_interval >> 16) << 1) | (next < 0 ? 1 : 0)))
        next |= 0x8000;
      else
        _counter = 1;
      regCMP(RTC.CMP + next);
    }

    static void regCTRLA(uint8_t value)
    {
      while (RTC.STATUS & RTC_CTRLABUSY_bm)
        continue;
      RTC.CTRLA = value;
    }

    static void regCNT(uint16_t value)
    {
      while (RTC.STATUS & RTC_CNTBUSY_bm)
        continue;
      RTC.CNT = value;
    }

    static void regPER(uint16_t value)
    {
      while (RTC.STATUS & RTC_PERBUSY_bm)
        continue;
      RTC.PER = value;
    }

    static void regCMP(uint16_t value)
    {
      while (RTC.STATUS & RTC_CMPBUSY_bm)
        continue;
      RTC.CMP = value;
    }

    static void regPITCTRLA(uint8_t value)
    {
      while (RTC.PITSTATUS & RTC_CTRLBUSY_bm)
        continue;
      RTC.PITCTRLA = value;
    }
};

#endif
