/*
  avr8_spi.h - SPI Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_clock.h"

#if   defined(SPI1)
#define SPI_MAX_CH 2
#elif defined(SPI0)
#define SPI_MAX_CH 1
#else
#define SPI_MAX_CH 0
#endif

class Spi
{
  public:

    /* SPI Mode select */
    typedef enum
    {
      MODE_0 = SPI_MODE_0_gc,  /* SPI Mode 0 */
      MODE_1 = SPI_MODE_1_gc,  /* SPI Mode 1 */
      MODE_2 = SPI_MODE_2_gc,  /* SPI Mode 2 */
      MODE_3 = SPI_MODE_3_gc,  /* SPI Mode 3 */
    } MODE;

    /* Prescaler select */
    typedef enum
    {
      PRESC_DIV2   = SPI_PRESC_DIV4_gc  | SPI_CLK2X_bm,  /* System Clock / 2 */
      PRESC_DIV4   = SPI_PRESC_DIV4_gc,                  /* System Clock / 4 */
      PRESC_DIV8   = SPI_PRESC_DIV16_gc | SPI_CLK2X_bm,  /* System Clock / 8 */
      PRESC_DIV16  = SPI_PRESC_DIV16_gc,                 /* System Clock / 16 */
      PRESC_DIV32  = SPI_PRESC_DIV64_gc | SPI_CLK2X_bm,  /* System Clock / 32 */
      PRESC_DIV64  = SPI_PRESC_DIV64_gc,                 /* System Clock / 64 */
      PRESC_DIV128 = SPI_PRESC_DIV128_gc,                /* System Clock / 128 */
    } PRESC;

    Spi(SPI_t& spi) : _spi(&spi), _ch((&spi - &SPI0) / sizeof(spi))
    {
      _instances[_ch] = this;
    }

    /* virtual */ ~Spi(void)
    {
      end();
      _instances[_ch] = 0;
    }

    bool begin(uint32_t frq, MODE mode = MODE_3)
    {
      if (frq)
      {
        static const uint8_t CLKDIV[] = { 2, 4, 8, 16, 32, 64, 128 };
        static const PRESC   PRESCS[] = { PRESC_DIV2, PRESC_DIV4, PRESC_DIV8, PRESC_DIV16, PRESC_DIV32, PRESC_DIV64, PRESC_DIV128 };          
        uint32_t div = Clock::frequency() / frq;
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          if (div <= CLKDIV[i])
          {
            begin(PRESCS[i], mode);
            return true;
          }
        }
        begin(PRESCS[sizeof(PRESCS) / sizeof(PRESCS[0]) - 1], mode);
      }
      return false;
    }

    void begin(PRESC presc, MODE mode)
    {
      end();
      _spi->INTCTRL = 0;
      _spi->CTRLB   = SPI_SSD_bm | mode;
      _spi->CTRLA   = SPI_MASTER_bm | presc | SPI_ENABLE_bm;
    }

    void end(void)
    {
      _spi->CTRLA = 0;
    }

    void lsbfirst(bool enable = true)
    {
      _spi->CTRLA = (_spi->CTRLA & ~SPI_DORD_bm) | (enable ? SPI_DORD_bm : 0);
    }

    uint8_t xfer(uint8_t data)
    {
      _spi->DATA = data;
      while ((_spi->INTFLAGS & SPI_IF_bm) == 0)
        continue;
      return _spi->DATA;
    }

    uint8_t read(void)
    {
      return xfer(0xFF);
    }

    void write(uint8_t data)
    {
      xfer(data);
    }

    void xfer(const void *wbuf, void *rbuf, size_t len)
    {
      const uint8_t* wp = (uint8_t*)wbuf;
      uint8_t* rp = (uint8_t*)rbuf;
      while (len)
      {
        uint8_t data = xfer(wp ? *wp++ : 0xFF);
        if (rp)
          *rp++ = data;
      }
    }

    void read(void *buf, size_t len)
    {
      xfer(0, buf, len);
    }

    void write(const void *buf, size_t len)
    {
      xfer(buf, 0, len);
    }

  protected:

    static Spi* _instances[SPI_MAX_CH];

    SPI_t*     _spi;
    uint8_t    _ch;
};
