/*
  avr8_task.cpp - Task Library for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#include <stdlib.h>
#if !defined(alloca)
  #include <alloca.h>
#endif
#include "avr8_task.h"

#if CONFIG_TASK_USE

Task::tcb_t *Task::_disp;
Task::tcb_t *Task::_curr;
Task::tcb_t *Task::_head;
Task::tcb_t *Task::_tail[];
size_t       Task::_used;

void Task::begin(size_t size)
{
  /* Registration of main task */
  static tcb_t tcb;
  _curr = &tcb;
  _used = size;
  AlarmTimer.begin();
}

bool Task::start(func_t func, void *arg, int size, void *stack)
{
  tcb_t *tcb;
  if (size < 0)
  {
    /* Use the current stack area */
    size_t n = - size;
    size  = _used;
    stack = 0;
    _used += n;
  }
  else if (stack)
  {
    /* Set the stack area to the variable area */
    size  = (char *)&tcb - (char *)stack - size + sizeof(*tcb);
    stack = 0;
  }
  else
  {
    /* Dynamically allocate stack area */
    stack = malloc(size);
    if (!stack)
      return false;
    size = (char *)&tcb - (char *)stack - size + sizeof(*tcb);
  }
  tcb = (tcb_t *)alloca(size);
  tcb->alloc = stack;
  tcb->arg   = arg;
  tcb->start = func;
  enqueue0(tcb);
  if (setjmp(tcb->context))
  {
    _curr = _disp;
    _curr->start(_curr->arg);
    stop();
  }
  return true;
}

void Task::stop(void)
{
  if (_curr->start)
  {
    if (_curr->alloc)
      free(_curr->alloc);
    dispatch();
  }
}

void Task::enqueue0(tcb_t *tcb)
{
  sreg_t state = disableAndSaveInterrupts();
  if (_tail[0])
  {
    tcb->next = _tail[0]->next;
    _tail[0]  = (_tail[0]->next = tcb);
  }
  else
  {
    tcb->next = _head;
    _tail[0]  = (_head = tcb);
  }
  restoreInterrupts(state);
}

void Task::enqueue1(tcb_t *tcb)
{
  sreg_t state = disableAndSaveInterrupts();
  (_tail[1] = *(_tail[1] ? &_tail[1]->next : (_tail[0] ? &_tail[0]->next : &_head)) = tcb)->next = 0;
  restoreInterrupts(state);
}

Task::tcb_t *Task::dequeue(void)
{
  tcb_t *tcb;
  sreg_t state = disableAndSaveInterrupts();
  if ((tcb = _head))
  {
    if (_head == _tail[0])
      _tail[0] = 0;
    else if (_head == _tail[1])
      _tail[1] = 0;
    _head = _head->next;
  }
  restoreInterrupts(state);
  return tcb;
}

size_t Task::current(void)
{
  return (size_t)_curr;
}

bool Task::yield(void)
{
  AlarmTimer.handle();
  enqueue1(_curr);
  switching();
  return !_curr->start;
}

void Task::switching(void)
{
  if (!setjmp(_curr->context))
    dispatch();
  _curr = _disp;
}

void Task::dispatch(void)
{
  while (1)
  {
    if ((_disp = dequeue()))
      longjmp(_disp->context, 1);
    idle();
    AlarmTimer.handle();
  }
}

void Task::sleep(int32_t tick, uint8_t busyLoop)
{
  uint32_t t = Timer.read();
  if (tick > busyLoop)
  {
    Sync sync;
    sync.sleep(tick -= busyLoop);
    t += tick;
  }
  if (busyLoop)
  {
    while (Timer.read() - t < busyLoop)
      continue;
  }
}

void Task::delay(uint32_t tick, uint8_t busyLoop)
{
  uint32_t t = Timer.read();
  if (tick > busyLoop)
  {
    tick -= busyLoop;
    while (Timer.read() - t < tick)
      yield();
    t += tick;
  }
  if (busyLoop)
  {
    while (Timer.read() - t < busyLoop)
      continue;
  }
}

Sync::Sync(void) : _head(0), _tail(0)
{
}

bool Sync::timeup(Alarm& alarm)
{
  if (((Sync *)alarm.param(0))->wakeup((size_t)alarm.param(1)))
    *(bool *)alarm.param(2) = false;
  return false;
}

bool Sync::sleep(ctrl_t& ctrl)
{
  tcb_t *tcb = (tcb_t *)current();
  (_tail = *(_tail ? &_tail->next : &_head) = tcb)->next = 0;
  restoreInterrupts(ctrl.state);
  ctrl.alarm.param(0, this);
  ctrl.alarm.param(1, tcb);
  ctrl.alarm.param(2, &ctrl.retval);
  ctrl.alarm.handler(timeup);
  AlarmTimer.start(ctrl.alarm);
  switching();
  return ctrl.retval;
}

bool Sync::sleep(int32_t timeout)
{
  if (timeout == 0)
    return false;
  ctrl_t ctrl;
  ctrl.alarm.interval(timeout);
  ctrl.retval = true;
  ctrl.state  = disableAndSaveInterrupts();
  return sleep(ctrl);
}

size_t Sync::wakeup(size_t task)
{
  tcb_t *tcb = 0;
  sreg_t state = disableAndSaveInterrupts();
  if (task)
  {
    for (tcb_t **p = &_head; *p; p = &(*p)->next)
    {
      if (*p == (tcb_t *)task)
      {
        if (!(*p = (*p)->next))
          _tail = (p == &_head ? 0 : (tcb_t *)p);
        tcb = (tcb_t *)task;
        break;
      }
    }
  }
  else
    _head = ((tcb = _head) != _tail ? _head->next : _tail = 0);
  restoreInterrupts(state);
  if (tcb)
  {
    AlarmTimer.cancel(1, tcb);
    enqueue0(tcb);
  }
  return (size_t)tcb;
}

Mutex::Mutex(void) : _owner(0)
{
}

bool Mutex::lock(int32_t timeout)
{
  ctrl_t ctrl;
  ctrl.alarm.interval(timeout);
  ctrl.retval = true;
  ctrl.state  = disableAndSaveInterrupts();
  size_t task = current();
  if (!_owner)
  {
    _owner = task;
    _count = 1;
  }
  else if (_owner == task)
    ++_count;
  else if (timeout == 0)
    ctrl.retval = false;
  else
    return sleep(ctrl);
  restoreInterrupts(ctrl.state);
  return ctrl.retval;
}

void Mutex::unlock(void)
{
  sreg_t state = disableAndSaveInterrupts();
  if (_owner == current())
  {
    if (--_count == 0)
    {
      _owner = wakeup();
      _count = 1;
    }
  }
  restoreInterrupts(state);
}

Sem::Sem(size_t limit, size_t count) : _limit(limit), _count(count), _wait(0)
{
}

bool Sem::wait(size_t count, int32_t timeout)
{
  ctrl_t ctrl;
  ctrl.alarm.interval(timeout);
  ctrl.retval = true;
  ctrl.state  = disableAndSaveInterrupts();
  if ((count == 0) || (count > _limit))
    ctrl.retval = false;
  else if (!_wait && (_count >= count))
    _count -= count;
  else if (timeout == 0)
    ctrl.retval = false;
  else
  {
    ++_wait;
    ((tcb_t *)current())->arg = (void *)count;
    sleep(ctrl);
    disableAndSaveInterrupts();
    --_wait;
    while (_head)
    {
      count = (size_t)_head->arg;
      if (_count < count)
        break;
      _count -= count;
      wakeup();
    }
  }
  restoreInterrupts(ctrl.state);
  return ctrl.retval;
}

void Sem::post(size_t count)
{
  sreg_t state = disableAndSaveInterrupts();
  if (count && (count + _count <= _limit))
  {
    _count += count;
    if (_head)
    {
      count = (size_t)_head->arg;
      if (_count >= count)
      {
        _count -= count;
        wakeup();
      }
    }
  }
  restoreInterrupts(state);
}

Event::Event(void) : Sem(1)
{
}

bool Event::wait(int32_t timeout)
{
  return Sem::wait(1, timeout);
}

void Event::signal(void)
{
  Sem::post();
}

#endif
