/*
  avr8_task.h - Task Library for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>
#include <setjmp.h>
#include "avr8_alarm.h"

#if CONFIG_TASK_USE

#define TASK_STACK_SIZE(n) ((n) + Task::TCB_SIZE)

class Task : public TimerClass
{
  public:

    typedef void (*func_t)(void *arg);

  protected:

    typedef struct tcb
    {
      struct tcb *next;
      jmp_buf context;
      void *alloc;
      void *arg;
      func_t start;
    } tcb_t;

    typedef struct
    {
      Alarm  alarm;
      sreg_t state;
      bool   retval;
    } ctrl_t;

    static void enqueue0(tcb_t *tcb);
    static void enqueue1(tcb_t *tcb);
    static tcb_t *dequeue(void);
    static void switching(void);
    static void dispatch(void);

  public:

    static const int TCB_SIZE = sizeof(tcb_t);

    static void begin(size_t size = TASK_STACK_SIZE(CONFIG_TASK_STACK_SIZE));
    static bool start(func_t func, void *arg = 0, int size = TASK_STACK_SIZE(CONFIG_TASK_STACK_SIZE), void *stack = 0);
    static void stop(void);
    static bool yield(void);
    static size_t current(void);
    static void sleep(int32_t tick, uint8_t busyLoop = 0);
    static void delay(uint32_t tick, uint8_t busyLoop = 0);

  private:

    static tcb_t *_disp;
    static tcb_t *_curr;
    static tcb_t *_head;
    static tcb_t *_tail[2];
    static size_t _used;
};

class Sync : protected Task
{
  public:

    Sync(void);
    bool sleep(int32_t timeout = -1);
    size_t wakeup(size_t task = 0);

  protected:

    bool sleep(ctrl_t& ctrl);
    static bool timeup(Alarm& alarm);

    tcb_t *_head;
    tcb_t *_tail;
};

class Mutex : protected Sync
{
  public:

    Mutex(void);
    bool lock(int32_t timeout = -1);
    void unlock(void);

  protected:

    size_t _owner;
    size_t _count;
};

class Sem : protected Sync
{
  public:

    Sem(size_t limit = (size_t)-1, size_t count = 0);
    bool wait(size_t count = 1, int32_t timeout = -1);
    void post(size_t count = 1);

  protected:

    size_t _limit;
    size_t _count;
    size_t _wait;
};

class Event : protected Sem
{
  protected:

    Event(void);
    bool wait(int32_t timeout = -1);
    void signal(void);
};

#endif
