/*
  avr8_tca.h - TCA Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include <string.h>
#include "avr8_config.h"
#include "avr8_clock.h"

#if   defined(TCA1)
#define TCA_MAX_CH 2
#elif defined(TCA0)
#define TCA_MAX_CH 1
#else
#define TCA_MAX_CH 0
#endif

#if TCA_MAX_CH

class TcaBase
{
  public:

    /* Derived Class Type */
    typedef enum
    {
      DCT_TCA,
      DCT_TCA8,
    } DCT;

    /* Compare Channel */
    typedef enum
    {
      CMP0,
      CMP1,
      CMP2,
    } CMP;

    /* Clock Selection select */
    typedef enum
    {
      CLKSEL_DIV1    = TCA_SINGLE_CLKSEL_DIV1_gc,    /* System Clock */
      CLKSEL_DIV2    = TCA_SINGLE_CLKSEL_DIV2_gc,    /* System Clock / 2 */
      CLKSEL_DIV4    = TCA_SINGLE_CLKSEL_DIV4_gc,    /* System Clock / 4 */
      CLKSEL_DIV8    = TCA_SINGLE_CLKSEL_DIV8_gc,    /* System Clock / 8 */
      CLKSEL_DIV16   = TCA_SINGLE_CLKSEL_DIV16_gc,   /* System Clock / 16 */
      CLKSEL_DIV64   = TCA_SINGLE_CLKSEL_DIV64_gc,   /* System Clock / 64 */
      CLKSEL_DIV256  = TCA_SINGLE_CLKSEL_DIV256_gc,  /* System Clock / 256 */
      CLKSEL_DIV1024 = TCA_SINGLE_CLKSEL_DIV1024_gc, /* System Clock / 1024 */
    } CLKSEL;

    /* Type of Callback Function */
    typedef void (*callback_t)(CMP cmp);

    TcaBase(DCT dct, TCA_t& tca) : _dct(dct), _tca(&tca), _ch((&tca - &TCA0) / sizeof(tca))
    {
      _instances[_ch] = this;
    }

    ~TcaBase(void)
    {
      end();
      _instances[_ch] = 0;
    }

    static void poll(void)
    {
#if !CONFIG_TCA_ISR
      for (uint8_t i = 0; i < TCA_MAX_CH; ++i)
      {
        TcaBase* tca = _instances[i];
        if (tca)
        {
          tca->isr_ovf();
          tca->isr_cmp(CMP0);
          tca->isr_cmp(CMP1);
          tca->isr_cmp(CMP2);
        }
      }
#endif
    }

  protected:

    static const uint16_t CLKDIV[8];
    static TcaBase* _instances[TCA_MAX_CH];

    DCT     const _dct;
    TCA_t*  const _tca;
    uint8_t const _ch;

  private:

#if CONFIG_TCA_ISR
    friend void tca_isr_ovf(uint8_t ch);
    friend void tca_isr_cmp(uint8_t ch, CMP cmp);
#endif

    void end(void);
    void isr_ovf(void);
    void isr_cmp(CMP cmp);
};

class Tca : public TcaBase
{
  public:

    /* Waveform generation mode select */
    typedef enum
    {
      WGMODE_NORMAL      = TCA_SINGLE_WGMODE_NORMAL_gc,      /* Normal Mode */
      WGMODE_FREQ        = TCA_SINGLE_WGMODE_FRQ_gc,         /* Frequency Generation Mode */
      WGMODE_SINGLESLOPE = TCA_SINGLE_WGMODE_SINGLESLOPE_gc, /* Single Slope PWM */
      WGMODE_DSTOP       = TCA_SINGLE_WGMODE_DSTOP_gc,       /* Dual Slope PWM, overflow on TOP */
      WGMODE_DSBOTH      = TCA_SINGLE_WGMODE_DSBOTH_gc,      /* Dual Slope PWM, overflow on TOP and BOTTOM */
      WGMODE_DSBOTTOM    = TCA_SINGLE_WGMODE_DSBOTTOM_gc,    /* Dual Slope PWM, overflow on BOTTOM */
    } WGMODE;

    /* Command select */
    typedef enum
    {
      CMD_NONE    = TCA_SINGLE_CMD_NONE_gc,     /* No Command */
      CMD_UPDATE  = TCA_SINGLE_CMD_UPDATE_gc,   /* Force Update */
      CMD_RESTART = TCA_SINGLE_CMD_RESTART_gc,  /* Force Restart */
      CMD_RESET   = TCA_SINGLE_CMD_RESET_gc,    /* Force Hard Reset */
    } CMD;

#if defined(TCA_SINGLE_EVACT_gm)
    /* Event Action select */
    typedef enum
    {
      EVACTA_CNT_POSEDGE = TCA_SINGLE_EVACT_POSEDGE_gc,  /* Count on positive edge event */
      EVACTA_CNT_ANYEDGE = TCA_SINGLE_EVACT_ANYEDGE_gc,  /* Count on any edge event */
      EVACTA_CNT_HIGHLVL = TCA_SINGLE_EVACT_HIGHLVL_gc,  /* Count on prescaled clock while event line is 1. */
      EVACTA_UPDOWN      = TCA_SINGLE_EVACT_UPDOWN_gc,   /* Count on prescaled clock. Event controls count direction. Up-count when event line is 0, down-count when event line is 1. */
    } EVACTA;
#endif

#if defined(TCA_SINGLE_EVACTA_gm)
    /* Event Action A select */
    typedef enum
    {
      EVACTA_CNT_POSEDGE = TCA_SINGLE_EVACTA_CNT_POSEDGE_gc,  /* Count on positive edge event */
      EVACTA_CNT_ANYEDGE = TCA_SINGLE_EVACTA_CNT_ANYEDGE_gc,  /* Count on any edge event */
      EVACTA_CNT_HIGHLVL = TCA_SINGLE_EVACTA_CNT_HIGHLVL_gc,  /* Count on prescaled clock while event line is 1. */
      EVACTA_UPDOWN      = TCA_SINGLE_EVACTA_UPDOWN_gc,       /* Count on prescaled clock. Event controls count direction. Up-count when event line is 0, down-count when event line is 1. */
    } EVACTA;
#endif

#if defined(TCA_SINGLE_EVACTB_gm)
    /* Event Action B select */
    typedef enum
    {
      EVACTB_NONE            = TCA_SINGLE_EVACTB_NONE_gc,             /* No Action */
      EVACTB_UPDOWN          = TCA_SINGLE_EVACTB_UPDOWN_gc,           /* Count on prescaled clock. Event controls count direction. Up-count when event line is 0, down-count when event line is 1. */
      EVACTB_RESTART_POSEDGE = TCA_SINGLE_EVACTB_RESTART_POSEDGE_gc,  /* Restart counter at positive edge event */
      EVACTB_RESTART_ANYEDGE = TCA_SINGLE_EVACTB_RESTART_ANYEDGE_gc,  /* Restart counter on any edge event */
      EVACTB_RESTART_HIGHLVL = TCA_SINGLE_EVACTB_RESTART_HIGHLVL_gc,  /* Restart counter while event line is 1. */
    } EVACTB;
#endif

    Tca(TCA_t& tca) : TcaBase(DCT_TCA, tca), _mhz(0)
    {
    }

    void begin(bool micros = true)
    {
      uint8_t mhz = Clock::frequency() / 1000000UL;
      uint8_t div = 0;
      if (micros)
      {
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          if (mhz <= CLKDIV[i])
          {
            if (mhz == CLKDIV[i])
            {
              div = i;
              mhz = 1;
            }
            break;
          }
        }
      }
      begin((CLKSEL)(CLKSEL_DIV1 + (div << TCA_SINGLE_CLKSEL_gp)), WGMODE_NORMAL, 0xFFFF, mhz);
    }

    bool begin(uint32_t frq, WGMODE mode)
    {
      if (frq)
      {
        uint32_t clk = Clock::frequency();
        if ((mode == WGMODE_FREQ) || (mode >= WGMODE_DSTOP))
          clk >>= 1;
        uint16_t div = (clk / frq + 0xFFFF) >> 16;
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          if (div <= CLKDIV[i])
          {
            begin((CLKSEL)(CLKSEL_DIV1 + (i << TCA_SINGLE_CLKSEL_gp)), mode, clk / CLKDIV[i] / frq - 1);
            return true;
          }
        }
        begin((CLKSEL)(CLKSEL_DIV1 + ((sizeof(CLKDIV) / sizeof(CLKDIV[0]) - 1) << TCA_SINGLE_CLKSEL_gp)), mode, 0xFFFF);
      }
      return false;
    }

    void begin(CLKSEL clksel, WGMODE mode, uint16_t period, uint8_t mhz = 0)
    {
      end();
      _mhz      = mhz;
      _min      = us2tick(24);
      _overflow = 0;
      memset((void*)_counter , 0, sizeof(_counter ));
      memset((void*)_interval, 0, sizeof(_interval));
      memset((void*)_callback, 0, sizeof(_callback));
      memset((void*)_duty    , 0, sizeof(_duty    ));
      _tca->SINGLE.CTRLD    = 0;
      _tca->SINGLE.CTRLC    = 0;
      _tca->SINGLE.CTRLB    = mode;
      _tca->SINGLE.CTRLECLR = 0xFF;
      _tca->SINGLE.EVCTRL   = 0;
#if CONFIG_TCA_ISR
      _tca->SINGLE.INTCTRL  = _mhz ? TCA_SINGLE_OVF_bm : 0;
#else
      _tca->SINGLE.INTCTRL  = 0;
#endif
      _tca->SINGLE.INTFLAGS = 0xFF;
      _tca->SINGLE.DBGCTRL  = 0;
      _tca->SINGLE.CNT      = 0;
      _tca->SINGLE.PER      = period;
      _tca->SINGLE.CMP0     = period;
      _tca->SINGLE.CMP1     = 0;
      _tca->SINGLE.CMP2     = 0;
      _tca->SINGLE.CTRLA    = clksel;
    }

    void end(void)
    {
      _tca->SINGLE.CTRLA = 0;
      control(CMD_RESET);
    }

    uint32_t frquency(void)
    {
      return Clock::frequency() / CLKDIV[(_tca->SINGLE.CTRLA & TCA_SINGLE_CLKSEL_gm) >> TCA_SINGLE_CLKSEL_gp];
    }

#if defined(TCA_SINGLE_RUNSTDBY_bm)
    void runstdby(bool enable = true)
    {
      _tca->SINGLE.CTRLA = (_tca->SINGLE.CTRLA & ~TCA_SINGLE_RUNSTDBY_bm) | (enable ? TCA_SINGLE_RUNSTDBY_bm : 0);
    }
#endif

    void dbgctrl(bool dbgrun = true)
    {
      _tca->SINGLE.DBGCTRL = dbgrun ? TCA_SINGLE_DBGRUN_bm : 0;
    }

    void run(bool enable = true)
    {
      _tca->SINGLE.CTRLA = (_tca->SINGLE.CTRLA & ~TCA_SINGLE_ENABLE_bm) | (enable ? TCA_SINGLE_ENABLE_bm : 0);
    }

    void output(CMP cmp, bool enable = true, bool ov = false)
    {
      uint8_t ovm = TCA_SINGLE_CMP0OV_bm << cmp;
      uint8_t enm = TCA_SINGLE_CMP0EN_bm << cmp;
      _tca->SINGLE.CTRLC = (_tca->SINGLE.CTRLC & ~ovm) | (ov     ? ovm : 0);
      _tca->SINGLE.CTRLB = (_tca->SINGLE.CTRLB & ~enm) | (enable ? enm : 0);
    }

    void compare(CMP cmp, uint16_t value)
    {
      (&_tca->SINGLE.CMP0)[cmp] = value;
    }

    void alupd(bool enable = true)
    {
      _tca->SINGLE.CTRLB = (_tca->SINGLE.CTRLB & ~TCA_SINGLE_ALUPD_bm) | (enable ? TCA_SINGLE_ALUPD_bm : 0);
    }

    void control(CMD cmd, bool lupd = false, bool dir = false)
    {
      if (dir)
        _tca->SINGLE.CTRLESET = TCA_SINGLE_DIR_bm;
      else
        _tca->SINGLE.CTRLECLR = TCA_SINGLE_DIR_bm;
      if (lupd)
        _tca->SINGLE.CTRLESET = TCA_SINGLE_LUPD_bm;
      else
        _tca->SINGLE.CTRLECLR = ~TCA_SINGLE_LUPD_bm;
      _tca->SINGLE.CTRLESET = cmd;
    }

    void eventa(bool enable, EVACTA evact)
    {
#if defined(TCA_SINGLE_EVACTA_gm)
      _tca->SINGLE.EVCTRL = (_tca->SINGLE.EVCTRL & ~(TCA_SINGLE_EVACTA_gm | TCA_SINGLE_CNTAEI_bm)) | evact | (enable ? TCA_SINGLE_CNTAEI_bm : 0);
#else
      _tca->SINGLE.EVCTRL = (_tca->SINGLE.EVCTRL & ~(TCA_SINGLE_EVACT_gm | TCA_SINGLE_CNTEI_bm)) | evact | (enable ? TCA_SINGLE_CNTEI_bm : 0);
#endif
    }

#if defined(TCA_SINGLE_EVACTB_gm)
    void eventb(bool enable, EVACTB evact)
    {
      _tca->SINGLE.EVCTRL = (_tca->SINGLE.EVCTRL & ~(TCA_SINGLE_EVACTB_gm | TCA_SINGLE_CNTBEI_bm)) | evact | (enable ? TCA_SINGLE_CNTBEI_bm : 0);
    }
#endif

    void pwmduty(CMP cmp, uint16_t duty)
    {
      if ((_tca->SINGLE.CTRLB & TCA_SINGLE_WGMODE_gm) >= TCA_SINGLE_WGMODE_SINGLESLOPE_gc)
        (&_tca->SINGLE.CMP0)[cmp] = (uint32_t)_tca->SINGLE.PER * (_duty[cmp] = duty) / 0xFFFF;
    }

    void period(uint16_t value)
    {
      if ((_tca->SINGLE.CTRLB & TCA_SINGLE_WGMODE_gm) == TCA_SINGLE_WGMODE_FRQ_gc)
        _tca->SINGLE.CMP0 = value;
      else if (_mhz == 0)
      {
        _tca->SINGLE.PER = value;
        if ((_tca->SINGLE.CTRLB & TCA_SINGLE_WGMODE_gm) >= TCA_SINGLE_WGMODE_SINGLESLOPE_gc)
        {
          pwmduty(CMP0, _duty[CMP0]);
          pwmduty(CMP1, _duty[CMP1]);
          pwmduty(CMP2, _duty[CMP2]);
        }
      }
    }

    void callback(CMP cmp, callback_t func)
    {
      _callback[cmp] = func;
    }

    void interval(CMP cmp, uint32_t tick, bool fromnow = false)
    {
      _interval[cmp] = tick <= _min ? _min : tick;
      if (fromnow)
      {
        (&_tca->SINGLE.CMP0)[cmp] = _tca->SINGLE.CNT;
        control(CMD_UPDATE);
      }
    }

    void interrupt(CMP cmp, uint32_t tick, bool restart = false)
    {
      if (_mhz)
      {
        uint8_t mask = TCA_SINGLE_CMP0_bm << cmp;
        if (tick)
        {
          interval(cmp, tick, true);
          if (restart || ((_tca->SINGLE.INTCTRL & mask) == 0))
          {
            setup(cmp);
#if CONFIG_TCA_ISR
            _tca->SINGLE.INTCTRL |= (_tca->SINGLE.INTFLAGS = mask);
#else
            _tca->SINGLE.INTFLAGS = mask;
#endif
            
          }
        }
        else
        {
          _tca->SINGLE.INTCTRL &= ~mask;
        }
      }
    }

    uint32_t read(void)
    {
      union
      {
        uint32_t d;
        struct { uint16_t l, h; } w;
      } x;
      do
      {
        x.w.h = _overflow;
        x.w.l = _tca->SINGLE.CNT;
#if CONFIG_TCA_ISR
        if ((SREG & SREG_I) == 0)
#endif
          isr_ovf();
      }
      while (x.w.h != _overflow);
      return x.d;
    }

    void delay(uint32_t tick)
    {
      if (_mhz)
      {
        for (uint32_t t = read(); read() - t < tick; )
          yield();
      }
    }

    uint32_t tick2us(uint32_t tick)
    {
      return _mhz <= 1 ? tick : tick / _mhz;
    }    

    uint32_t us2tick(uint32_t us)
    {
      return _mhz <= 1 ? us : us * _mhz;
    }    

  protected:

    uint8_t           _mhz;
    uint16_t          _min;
    volatile uint16_t _overflow;
    volatile uint16_t _counter [3];
    uint32_t          _interval[3];
    callback_t        _callback[3];
    uint16_t          _duty[3];

    friend class TcaBase;

    void isr_ovf(void)
    {
      if ((_tca->SINGLE.INTFLAGS = (_tca->SINGLE.INTFLAGS & TCA_SINGLE_OVF_bm)))
      {
        if (_mhz)
          ++_overflow;
      }
    }

    void isr_cmp(CMP cmp)
    {
      if ((_tca->SINGLE.INTFLAGS = (_tca->SINGLE.INTFLAGS & (TCA_SINGLE_CMP0_bm << cmp))))
      {
        if (_mhz)
        {
          if (--_counter[cmp])
          {
            (&_tca->SINGLE.CMP0)[cmp] += 0x8000;
            control(CMD_UPDATE);
          }
          else
          {
            if (_callback[cmp])
              _callback[cmp](cmp);
            setup(cmp);
          }
        }
      }
    }

    void setup(CMP cmp)
    {
      uint32_t tick = _interval[cmp];
      int16_t  next = (int16_t)tick;
      if ((_counter[cmp] = ((uint16_t)(tick >> 16) << 1) | (next < 0 ? 1 : 0)))
        next |= 0x8000;
      else
        _counter[cmp] = 1;
      (&_tca->SINGLE.CMP0)[cmp] += next;
      control(CMD_UPDATE);
    }
};

class Tca8 : public TcaBase
{
  public:

    /* PWM Channel */
    typedef enum
    {
      PWM0,
      PWM1,
    } PWM;

    /* Command select */
    typedef enum
    {
      CMD_NONE    = TCA_SPLIT_CMD_NONE_gc,     /* No Command */
      CMD_RESTART = TCA_SPLIT_CMD_RESTART_gc,  /* Force Restart */
      CMD_RESET   = TCA_SPLIT_CMD_RESET_gc,    /* Force Hard Reset */
    } CMD;

#if defined(TCA_SPLIT_CMDEN_gm)
    /* Command Enable select (SPLIT Mode) */
    typedef enum
    {
      CMDEN_NONE = TCA_SPLIT_CMDEN_NONE_gc,  /* None */
      CMDEN_BOTH = TCA_SPLIT_CMDEN_BOTH_gc,  /* Both low byte and high byte counter */
    } CMDEN;
#endif

    Tca8(TCA_t& tca) : TcaBase(DCT_TCA8, tca), _timerovf(0)
    {
    }

    bool begin(void)
    {
      CLKSEL sel;
      switch (Clock::frequency())
      {
        default: return false;
        case 16000000UL: sel = CLKSEL_DIV64; break;
        case  4000000UL: sel = CLKSEL_DIV16; break;
        case  2000000UL: sel = CLKSEL_DIV8 ; break;
        case  1000000UL: sel = CLKSEL_DIV4 ; break;
      }
      begin(sel, 0xFF, true);
      return true;
    }

    bool begin(uint32_t frq)
    {
      if (frq)
      {
        uint32_t clk = Clock::frequency();
        uint32_t div = (clk / frq + 0xFF) >> 8;
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          if (div <= CLKDIV[i])
          {
            begin((CLKSEL)(CLKSEL_DIV1 + (i << TCA_SPLIT_CLKSEL_gp)), clk / CLKDIV[i] / frq - 1);
            return true;
          }
        }
        begin((CLKSEL)(CLKSEL_DIV1 + ((sizeof(CLKDIV) / sizeof(CLKDIV[0]) - 1) << TCA_SPLIT_CLKSEL_gp)), 0xFF);
      }
      return false;
    }

    void begin(CLKSEL clksel, uint8_t per, bool timer = false)
    {
      end();
      _timerovf = timer ? TCA_SPLIT_LUNF_bm : 0;
      _min      = us2tick(24);
      _overflow = 0;
      memset((void*)_counter , 0, sizeof(_counter ));
      memset((void*)_interval, 0, sizeof(_interval));
      memset((void*)_callback, 0, sizeof(_callback));
      memset((void*)_duty    , 0, sizeof(_duty    ));
      _tca->SPLIT.CTRLD    = TCA_SPLIT_SPLITM_bm;
      _tca->SPLIT.CTRLC    = 0;
      _tca->SPLIT.CTRLB    = 0;
      _tca->SPLIT.CTRLECLR = 0xFF;
      _tca->SPLIT.INTFLAGS = 0xFF;
#if CONFIG_TCA_ISR
      _tca->SPLIT.INTCTRL  = timer ? TCA_SPLIT_LUNF_bm : 0;
#else
      _tca->SPLIT.INTCTRL  = 0;
#endif
      _tca->SPLIT.DBGCTRL  = 0;
      _tca->SPLIT.LCNT     = 0;
      _tca->SPLIT.HCNT     = 0;
      period(PWM0, per);
      period(PWM1, per);
      _tca->SPLIT.CTRLA    = clksel;
    }

    void end(void)
    {
      _tca->SPLIT.CTRLA = 0;
      control(CMD_RESET);
    }

    uint32_t frquency(void)
    {
      return Clock::frequency() / CLKDIV[(_tca->SPLIT.CTRLA & TCA_SPLIT_CLKSEL_gm) >> TCA_SPLIT_CLKSEL_gp];
    }

#if defined(TCA_SPLIT_RUNSTDBY_bm)
    void runstdby(bool enable = true)
    {
      _tca->SPLIT.CTRLA = (_tca->SPLIT.CTRLA & ~TCA_SPLIT_RUNSTDBY_bm) | (enable ? TCA_SPLIT_RUNSTDBY_bm : 0);
    }
#endif

    void dbgctrl(bool dbgrun = true)
    {
      _tca->SPLIT.DBGCTRL = dbgrun ? TCA_SPLIT_DBGRUN_bm : 0;
    }

    void run(bool enable = true)
    {
      _tca->SPLIT.CTRLA = (_tca->SPLIT.CTRLA & ~TCA_SPLIT_ENABLE_bm) | (enable ? TCA_SPLIT_ENABLE_bm : 0);
    }

    void output(PWM pwm, CMP cmp, bool enable = true, bool ov = false)
    {
      uint8_t ovm = (pwm == PWM0 ? TCA_SPLIT_LCMP0OV_bm : TCA_SPLIT_HCMP0OV_bm) << cmp;
      uint8_t enm = (pwm == PWM0 ? TCA_SPLIT_LCMP0EN_bm : TCA_SPLIT_HCMP0EN_bm) << cmp;
      _tca->SPLIT.CTRLC = (_tca->SPLIT.CTRLC & ~ovm) | (ov     ? ovm : 0);
      _tca->SPLIT.CTRLB = (_tca->SPLIT.CTRLB & ~enm) | (enable ? enm : 0);
    }

#if defined(TCA_SPLIT_CMDEN_gm)
    void control(CMD cmd, CMDEN cmden = CMDEN_NONE)
    {
      _tca->SPLIT.CTRLESET = cmd | cmden;
    }
#else
    void control(CMD cmd)
    {
      _tca->SPLIT.CTRLESET = cmd;
    }
#endif

    void pwmduty(PWM pwm, CMP cmp, uint8_t duty)
    {
      if ((pwm == PWM1) || (_timerovf == 0))
        (&_tca->SPLIT.LCMP0 + (cmp << 1))[pwm] = (uint16_t)(&_tca->SPLIT.LPER)[pwm] * (_duty[pwm][cmp] = duty) / 0xFF;
    }

    void period(PWM pwm, uint8_t value)
    {
      if ((pwm == PWM1) || (_timerovf == 0))
      {
        (&_tca->SPLIT.LPER)[pwm] = value;
        pwmduty(pwm, CMP0, _duty[pwm][CMP0]);
        pwmduty(pwm, CMP1, _duty[pwm][CMP1]);
        pwmduty(pwm, CMP2, _duty[pwm][CMP2]);
      }
    }

    void callback(CMP cmp, callback_t func)
    {
      _callback[cmp] = func;
    }

    void interval(CMP cmp, uint32_t tick, bool fromnow = false)
    {
      _interval[cmp] = tick <= _min ? _min : tick;
      if (fromnow)
        (&_tca->SPLIT.LCMP0)[cmp << 1] = _tca->SPLIT.LCNT;
    }

    void interrupt(CMP cmp, uint32_t tick, bool restart = false)
    {
      if (_timerovf)
      {
        uint8_t mask = TCA_SPLIT_LCMP0_bm << cmp;
        if (tick)
        {
          interval(cmp, tick, true);
          if (restart || ((_tca->SINGLE.INTCTRL & mask) == 0))
          {
            setup(cmp);
#if CONFIG_TCA_ISR
            _tca->SINGLE.INTCTRL |= _tca->SPLIT.INTFLAGS = mask;
#else
            _tca->SINGLE.INTFLAGS = mask;
#endif
          }
        }
        else
        {
          _tca->SINGLE.INTCTRL &= ~mask;
        }
      }
    }

    uint32_t read(void)
    {
      union
      {
        uint32_t d;
        struct
        {
          uint8_t  l;
          uint32_t h;
        } w;
      } x;
      do
      {
        x.w.h = _overflow;
        x.w.l = ~_tca->SPLIT.LCNT;
#if CONFIG_TCA_ISR
        if ((SREG & SREG_I) == 0)
#endif
          isr_ovf();
      }
      while (x.w.h != _overflow);
      return x.d;
    }

    void delay(uint32_t tick)
    {
      if (_timerovf)
      {
        for (uint32_t t = read(); read() - t < tick; )
          yield();
      }
    }

    uint32_t tick2us(uint32_t tick)
    {
      return tick << 2;
    }

    uint32_t us2tick(uint32_t us)
    {
      return us >> 2;
    }

  protected:

    uint8_t           _timerovf;
    uint16_t          _min;
    volatile uint32_t _overflow;
    volatile uint32_t _counter [3];
    uint32_t          _interval[3];
    callback_t        _callback[3];
    uint8_t           _duty[2][3];

    friend class TcaBase;

    void isr_ovf(void)
    {
      if ((_tca->SPLIT.INTFLAGS = (_tca->SPLIT.INTFLAGS & (TCA_SPLIT_LUNF_bm | TCA_SPLIT_HUNF_bm))) & _timerovf)
        ++_overflow;
    }

    void isr_cmp(CMP cmp)
    {
      if ((_tca->SPLIT.INTFLAGS = (_tca->SPLIT.INTFLAGS & (TCA_SPLIT_LCMP0_bm << cmp))))
      {
        if (_timerovf)
        {
          if (--_counter[cmp])
            (&_tca->SPLIT.LCMP0)[cmp << 1] -= 0x80;
          else
          {
            if (_callback[cmp])
              _callback[cmp](cmp);
            setup(cmp);
          }
        }          
      }
    }

    void setup(CMP cmp)
    {
      uint32_t tick = _interval[cmp];
      int8_t   next = (int8_t)tick;
      if ((_counter[cmp] = ((tick >> 8) << 1) | (next < 0 ? 1 : 0)))
        next |= 0x80;
      else
        _counter[cmp] = 1;
      (&_tca->SPLIT.LCMP0)[cmp << 1] -= next;
    }
};

#endif
