/*
  avr8_tcb.h - TCB Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_tcb_dfp.h"
#include "avr8_clock.h"

#if   defined(TCB4)
#define TCB_MAX_CH 5
#elif defined(TCB3)
#define TCB_MAX_CH 4
#elif defined(TCB2)
#define TCB_MAX_CH 3
#elif defined(TCB1)
#define TCB_MAX_CH 2
#elif defined(TCB0)
#define TCB_MAX_CH 1
#else
#define TCB_MAX_CH 0
#endif

#if defined(TCB_CLKSEL_CLKDIV1_gc)
  #define TCB_CLKSEL_DIV1_gc TCB_CLKSEL_CLKDIV1_gc
#endif
#if defined(TCB_CLKSEL_CLKDIV2_gc)
  #define TCB_CLKSEL_DIV2_gc TCB_CLKSEL_CLKDIV2_gc
#endif
#if defined(TCB_CLKSEL_CLKTCA_gc)
  #define TCB_CLKSEL_TCA0_gc TCB_CLKSEL_CLKTCA_gc
#endif

class Tcb
{
  public:

    /* Clock Select select */
    typedef enum
    {
      CLKSEL_DIV1  = TCB_CLKSEL_DIV1_gc,     /* CLK_PER (No Prescaling) */
      CLKSEL_DIV2  = TCB_CLKSEL_DIV2_gc,     /* CLK_PER/2 (From Prescaler) */
#if defined(TCB_CLKSEL_TCA0_gc)
      CLKSEL_TCA0  = TCB_CLKSEL_TCA0_gc,     /* Use CLK_TCA from TCA0 */
#endif
#if defined(TCB_CLKSEL_TCA1_gc)
      CLKSEL_TCA1  = TCB_CLKSEL_TCA1_gc,     /* Use CLK_TCA from TCA1 */
#endif
#if defined(TCB_CLKSEL_TCE0_gc)
      CLKSEL_TCE0  = TCB_CLKSEL_TCE0_gc,    /* Use CLK_TCE from TCE0 */
#endif
#if defined(TCB_CLKSEL_EVENT_gc)
      CLKSEL_EVENT = TCB_CLKSEL_EVENT_gc,   /* Count on event edge */
#endif
    } CLKSEL;

    /* Timer Mode select */
    typedef enum
    {
      CNTMODE_INT     = TCB_CNTMODE_INT_gc,      /* Periodic Interrupt */
      CNTMODE_TIMEOUT = TCB_CNTMODE_TIMEOUT_gc,  /* Periodic Timeout */
      CNTMODE_CAPT    = TCB_CNTMODE_CAPT_gc,     /* Input Capture Event */
      CNTMODE_FRQ     = TCB_CNTMODE_FRQ_gc,      /* Input Capture Frequency measurement */
      CNTMODE_PW      = TCB_CNTMODE_PW_gc,       /* Input Capture Pulse-Width measurement */
      CNTMODE_FRQPW   = TCB_CNTMODE_FRQPW_gc,    /* Input Capture Frequency and Pulse-Width measurement */
      CNTMODE_SINGLE  = TCB_CNTMODE_SINGLE_gc,   /* Single Shot */
      CNTMODE_PWM8    = TCB_CNTMODE_PWM8_gc,     /* 8-bit PWM */
    } CNTMODE;

#if defined(TCB_CNTSIZE_gm)
    /* Counter Size select */
    typedef enum
    {
      CNTSIZE_16BITS = TCB_CNTSIZE_16BITS_gc,  /* 16-bit CNT. MAX=16'hFFFF */
      CNTSIZE_15BITS = TCB_CNTSIZE_15BITS_gc,  /* 15-bit CNT. MAX=16'h7FFF */
      CNTSIZE_14BITS = TCB_CNTSIZE_14BITS_gc,  /* 14-bit CNT. MAX=16'h3FFF */
      CNTSIZE_13BITS = TCB_CNTSIZE_13BITS_gc,  /* 13-bit CNT. MAX=16'h1FFF */
      CNTSIZE_12BITS = TCB_CNTSIZE_12BITS_gc,  /* 12-bit CNT. MAX=16'h0FFF */
      CNTSIZE_11BITS = TCB_CNTSIZE_11BITS_gc,  /* 11-bit CNT. MAX=16'h07FF */
      CNTSIZE_10BITS = TCB_CNTSIZE_10BITS_gc,  /* 10-bit CNT. MAX=16'h03FF */
      CNTSIZE_9BITS  = TCB_CNTSIZE_9BITS_gc,   /* 9-bit CNT. MAX=16'h01FF */
    } CNTSIZE;
#endif

#if defined(TCB_CNTSIZE_gm)
    /* Event Generation select */
    typedef enum
    {
      EVGEN_PULSE    = TCB_EVGEN_PULSE_gc,     /* Event is generated as pulse at compare match or capture */
      EVGEN_WAVEFORM = TCB_EVGEN_WAVEFORM_gc,  /* Event is generated as waveform for modes with waveform */
    } EVGEN;
#endif

    /* TCB.STATUS  bit masks and bit positions */
    typedef enum
    {
      STATUS_RUN = TCB_RUN_bm,  /* Run bit mask. */
    } STATUS;

    /* TCB.INTCTRL  bit masks and bit positions */
    typedef enum
    {
      INTFLAGS_CAPT = TCB_CAPT_bm,  /* Capture or Timeout bit mask. */
#if defined(TCB_OVF_bm)
      INTFLAGS_OVF  = TCB_OVF_bm,   /* Overflow bit mask. */
#endif
    } INTFLAGS;

    /* Type of Callback Function */
    typedef void (*callback_t)(INTFLAGS flags);

    Tcb(TCB_t& tcb) : _tcb(&tcb), _ch((&tcb - &TCB0) / sizeof(tcb)), _mhz(0), _period(0)
    {
      _instances[_ch] = this;
    }

    /* virtual */ ~Tcb(void)
    {
      end();
      _instances[_ch] = 0;
    }

    void begin(bool micros = true)
    {
      uint8_t mhz = Clock::frequency() / 1000000UL;
      CLKSEL  div = CLKSEL_DIV1;
      if (micros)
      {
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          CLKSEL  sel = (CLKSEL)(CLKSEL_DIV1 + (i << TCB_CLKSEL_gp));
          uint8_t idx = clksel(sel);
          if (idx < sizeof(CLKDIV) / sizeof(CLKDIV[0]))
          {
            if (mhz <= CLKDIV[idx])
            {
              if (mhz == CLKDIV[idx])
              {
                div = sel;
                mhz = 1;
              }
              break;
            }
          }
        }
      }
      begin(div, CNTMODE_INT, 0xFFFF, mhz);
    }

    bool begin(uint32_t frq, CNTMODE mode)
    {
      if (frq)
      {
        uint32_t clk = Clock::frequency();
        uint32_t div = mode == CNTMODE_PWM8 ? (clk / frq + 0xFF) >> 8 : (clk / frq + 0xFFFF) >> 16;
        CLKSEL   max_sel = CLKSEL_DIV2;
        uint16_t max_div = 2;
        for (uint8_t i = 0; i < sizeof(CLKDIV) / sizeof(CLKDIV[0]); ++i)
        {
          CLKSEL  sel = (CLKSEL)(CLKSEL_DIV1 + (i << TCB_CLKSEL_gp));
          uint8_t idx = clksel(sel);
          if (idx < sizeof(CLKDIV) / sizeof(CLKDIV[0]))
          {
            if (div <= CLKDIV[idx])
            {
              begin(sel, mode, clk / CLKDIV[idx] / frq - 1);
              return true;
            }
            if (max_div < CLKDIV[idx])
            {
              max_div = CLKDIV[idx];
              max_sel = sel;
            }            
          }
        }
        begin(max_sel, mode, 0xFFFF);
      }
      return false;
    }

    void begin(CLKSEL clksel, CNTMODE mode, uint16_t period, uint8_t mhz = 0)
    {
      end();
      _mhz           = mhz;
      _overflow      = 0;
      _callback      = 0;
      _tcb->CCMP     = _period = period;
      _tcb->CNT      = 0;
      _tcb->DBGCTRL  = 0;
      _tcb->INTFLAGS = TCB_CAPT_bm;
      _tcb->INTCTRL  = _mhz ? TCB_CAPT_bm : 0;
      _tcb->EVCTRL   = 0;
#if defined(TCB_CNTSIZE_gm)
      _tcb->CTRLC    = 0;
#endif
      _tcb->CTRLB    = mode;
      _tcb->CTRLA    = clksel;
    }

    void end(void)
    {
      _tcb->CTRLA = 0;
    }

    uint32_t frquency(void)
    {
      uint8_t idx = clksel((CLKSEL)(_tcb->CTRLA & TCB_CLKSEL_gm));
      return idx < sizeof(CLKDIV) / sizeof(CLKDIV[0]) ? Clock::frequency() / CLKDIV[idx] : 0;
    }

    void runstdby(bool enable = true)
    {
      _tcb->CTRLA = (_tcb->CTRLA & ~TCB_RUNSTDBY_bm) | (enable ? TCB_RUNSTDBY_bm : 0);
    }

    void dbgctrl(bool dbgrun = true)
    {
      _tcb->DBGCTRL = dbgrun ? TCB_DBGRUN_bm : 0;
    }

    void run(bool enable = true)
    {
      _tcb->CTRLA = (_tcb->CTRLA & ~TCB_ENABLE_bm) | (enable ? TCB_ENABLE_bm : 0);
    }

    void syncupd(bool enable = true)
    {
      _tcb->CTRLA = (_tcb->CTRLA & ~TCB_SYNCUPD_bm) | (enable ? TCB_SYNCUPD_bm : 0);
    }

#if defined(TCB_CASCADE_bm)
    void cascade(bool enable = true)
    {
      _tcb->CTRLA = (_tcb->CTRLA & ~TCB_CASCADE_bm) | (enable ? TCB_CASCADE_bm : 0);
    }
#endif

#if defined(TCB_CNTSIZE_gm)
    void cntsize(CNTSIZE size = CNTSIZE_16BITS)
    {
      _tcb->CTRLC = size;
    }
#endif

    void output(bool enable = true, uint8_t init = 0, bool async = false)
    {
#if defined(TCB_EVGEN_bm)
      _tcb->CTRLB = (_tcb->CTRLB & ~(TCB_EVGEN_bm | TCB_ASYNC_bm | TCB_CCMPINIT_bm | TCB_CCMPEN_bm))
#else
      _tcb->CTRLB = (_tcb->CTRLB & ~(TCB_ASYNC_bm | TCB_CCMPINIT_bm | TCB_CCMPEN_bm))
#endif
        | (enable ? TCB_CCMPEN_bm : 0) | (async ? TCB_ASYNC_bm : 0) | (init ? TCB_CCMPINIT_bm : 0);
    }

#if defined(TCB_EVGEN_bm)
    void evgen(bool enable = true)
    {
      _tcb->CTRLB = (_tcb->CTRLB & ~(TCB_EVGEN_bm | TCB_CCMPEN_bm)) | (enable ? TCB_EVGEN_bm : 0);
    }
#endif

    void event(bool capei = false, bool edge = false, bool filter = false)
    {
      _tcb->EVCTRL = (capei ? TCB_CAPTEI_bm : 0) | (edge ? TCB_EDGE_bm : 0) | (filter ? TCB_FILTER_bm : 0);
    }

    void pwmduty(uint8_t duty)
    {
      if ((_tcb->CTRLB & TCB_CNTMODE_gm) == TCB_CNTMODE_PWM8_gc)
      {
        union { uint8_t b[2]; uint16_t w; } x;
        x.w = (uint8_t)_period;
        x.b[1] = (x.w + (x.w < 0xFF)) * duty / 0xFF;
        _tcb->CCMP = x.w;
        x.b[1] = duty;
        _period = x.w;
      }
    }

    void period(uint16_t value)
    {
      if ((_tcb->CTRLB & TCB_CNTMODE_gm) == TCB_CNTMODE_PWM8_gc)
      {
        uint8_t duty = _period >> 8;
        _period = value;
        pwmduty(duty);
      }        
      else
        _tcb->CCMP = _period = value;
    }

    uint16_t capture(void)
    {
      return _tcb->CCMP;
    }

    void callback(callback_t func)
    {
      _callback = func;
    }

    void interrupt(bool enable)
    {
      if (enable)
#if CONFIG_TCB_ISR
        _tcb->INTCTRL = _tcb->INTFLAGS = TCB_CAPT_bm;
#else
        _tcb->INTFLAGS = TCB_CAPT_bm;
#endif
      else if (_mhz == 0)
        _tcb->INTCTRL = 0;
    }

    uint32_t read(void)
    {
      union
      {
        uint32_t d;
        struct { uint16_t l, h; } w;
      } x;
      do
      {
        x.w.h = _overflow;
        x.w.l = _tcb->CNT;
#if CONFIG_TCB_ISR
        if ((SREG & SREG_I) == 0)
#endif
          isr_capt();
      }
      while (x.w.h != _overflow);
      return x.d;
    }

    void delay(uint32_t tick)
    {
      if (_mhz)
      {
        for (uint32_t t = read(); read() - t < tick; )
          yield();
      }
    }

    uint32_t tick2us(uint32_t tick)
    {
      return _mhz <= 1 ? tick : tick / _mhz;
    }    

    uint32_t us2tick(uint32_t us)
    {
      return _mhz <= 1 ? us : us * _mhz;
    }

    static void poll(void)
    {
#if !CONFIG_TCB_ISR
      for (uint8_t i = 0; i < TCB_MAX_CH; ++i)
      {
        Tcb* tcb = _instances[i];
        if (tcb)
          tcb->isr_capt();
      }        
#endif      
    }

  protected:

    static const uint16_t CLKDIV[8];
    static Tcb* _instances[TCB_MAX_CH];

    TCB_t*             _tcb;
    uint8_t            _ch;
    uint8_t            _mhz;
    uint16_t           _period;
    volatile uint16_t _overflow;
    callback_t        _callback;

#if defined(CONFIG_TCB_ISR)
    friend void tcb_isr_capt(uint8_t ch);
#endif

    inline void isr_capt(void)
    {
      uint8_t flags = _tcb->INTFLAGS = _tcb->INTFLAGS;
      if (_mhz && (flags & TCB_CAPT_bm))
        ++_overflow;
      if (_callback)
      _callback((INTFLAGS)flags);
    }

    uint8_t clksel(CLKSEL clksel)
    {
      switch (clksel)
      {
        case TCB_CLKSEL_DIV1_gc:
        case TCB_CLKSEL_DIV2_gc:
          return clksel >> TCB_CLKSEL_gp;
#if defined(TCB_CLKSEL_TCA0_gc)
        case TCB_CLKSEL_TCA0_gc:
          if (TCA0.SINGLE.CTRLA & TCA_SINGLE_ENABLE_bm)
            return (TCA0.SINGLE.CTRLA & TCA_SINGLE_CLKSEL_gm) >> TCA_SINGLE_CLKSEL_gp;
          break;
#endif
#if defined(TCB_CLKSEL_TCA1_gc)
        case TCB_CLKSEL_TCA1_gc:
          if (TCA1.SINGLE.CTRLA & TCA_SINGLE_ENABLE_bm)
            return (TCA1.SINGLE.CTRLA & TCA_SINGLE_CLKSEL_gm) >> TCA_SINGLE_CLKSEL_gp;
          break;
#endif
#if defined(TCB_CLKSEL_TCE0_gc)
        case TCB_CLKSEL_TCE0_gc:
          if (TCE0.CTRLA & TCE_ENABLE_bm)
            return (TCE0.CTRLA & TCE_CLKSEL_gm) >> TCE_CLKSEL_gp;
          break;
#endif
        default:
          break;
      }
      return 0xFF;
    }
};
