/*
  avr8_timer.cpp - Timer Library for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#include "avr8_config.h"

#if CONFIG_TASK_USE || CONFIG_ALARM_USE

#include "avr8_timer.h"
#include "avr8_alarm.h"

#if CONFIG_ALARM_TIMER == TIMER_RTC
  Rtc Timer;
#elif CONFIG_ALARM_TIMER == TIMER_TCA
  Tca Timer(CONFIG_ALARM_TIMER_TCA);
#elif CONFIG_ALARM_TIMER == TIMER_TCA8
  Tca8 Timer(CONFIG_ALARM_TIMER_TCA);
#elif CONFIG_ALARM_TIMER == TIMER_TCB
  Tcb Timer(CONFIG_ALARM_TIMER_TCB);
#endif

/* RTC Timer */

#if CONFIG_ALARM_TIMER == TIMER_RTC

void AlarmClass_timeup(void)
{
  AlarmTimer.timeup();
}

bool TimerClass::setupAlarmTimer(void)
{
  Timer.begin();
  Timer.callback(AlarmClass_timeup);
  Timer.run();
  return true;
}

void TimerClass::startAlarmTimer(int32_t interval)
{
  Timer.interrupt(interval);
}

/* Tca Timer */

#elif CONFIG_ALARM_TIMER == TIMER_TCA

void AlarmClass_timeup(TcaBase::CMP cmp)
{
  (void)cmp;
  AlarmTimer.timeup();
}

bool TimerClass::setupAlarmTimer(void)
{
  Timer.begin();
  Timer.callback(CONFIG_ALARM_TIMER_TCA_CMP, AlarmClass_timeup);
  Timer.run();
#if CONFIG_TCA_ISR
  return true;
#else
  return false;
#endif
}

void TimerClass::startAlarmTimer(int32_t interval)
{
  Timer.interrupt(CONFIG_ALARM_TIMER_TCA_CMP, interval);
}

/* Tca8 Timer */

#elif CONFIG_ALARM_TIMER == TIMER_TCA8

void AlarmClass_timeup(TcaBase::CMP cmp)
{
  (void)cmp;
  AlarmTimer.timeup();
}

bool TimerClass::setupAlarmTimer(void)
{
  if (!Timer.begin())
    return false;
  Timer.callback(CONFIG_ALARM_TIMER_TCA_CMP, AlarmClass_timeup);
  Timer.run();
#if CONFIG_TCA_ISR
  return true;
#else
  return false;
#endif
}

void TimerClass::startAlarmTimer(int32_t interval)
{
  Timer.interrupt(CONFIG_ALARM_TIMER_TCA_CMP, interval);
}

/* Tcb Timer */

#elif CONFIG_ALARM_TIMER == TIMER_TCB

bool TimerClass::setupAlarmTimer(void)
{
  Timer.begin();
  Timer.run();
  return false;
}

void TimerClass::startAlarmTimer(int32_t interval)
{
  (void)interval;
}

#endif

sreg_t TimerClass::disableAndSaveInterrupts(void)
{
  sreg_t rv = SREG;
  cli();
  return rv;
}

void TimerClass::restoreInterrupts(sreg_t save)
{
  SREG = save;
}

void TimerClass::idle(void)
{
  sei();
}

#endif
