/*
  avr8_twi.h - TWI Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"
#include "avr8_clock.h"
#if CONFIG_RTC_USE
  #include "avr8_rtc.h"
#endif

#if   defined(TWI1)
#define TWI_MAX_CH 2
#elif defined(TWI0)
#define TWI_MAX_CH 1
#else
#define TWI_MAX_CH 0
#endif

#if !defined(CONFIG_TWI_BUS_TIMEOUT)
  #define CONFIG_TWI_BUS_TIMEOUT 200000
#endif

class Twi
{
  public:

    /* TWI Mode */
    typedef enum
    {
      MODE_100KHZ =  100000UL, /* Standard Mode */
      MODE_400KHZ =  400000UL, /* First Mode */
      MODE_1MHZ   = 1000000UL, /* First Mode Plus */
    } MODE;

    /* SDA Setup Time select */
    typedef enum
    {
      SDASETUP_4CYC = TWI_SDASETUP_4CYC_gc,  /* SDA setup time is 4 clock cycles */
      SDASETUP_8CYC = TWI_SDASETUP_8CYC_gc,  /* SDA setup time is 8 clock cycles */
    } SDASETUP;

    /* SDA Hold Time select */
    typedef enum
    {
      SDAHOLD_OFF   = TWI_SDAHOLD_OFF_gc,    /* SDA hold time off */
      SDAHOLD_50NS  = TWI_SDAHOLD_50NS_gc,   /* Typical 50ns hold time */
      SDAHOLD_300NS = TWI_SDAHOLD_300NS_gc,  /* Typical 300ns hold time */
      SDAHOLD_500NS = TWI_SDAHOLD_500NS_gc,  /* Typical 500ns hold time */
    } SDAHOLD;

    /* Inactive Bus Timeout select */
    typedef enum
    {
      TIMEOUT_DISABLED = TWI_TIMEOUT_DISABLED_gc,  /* Bus Timeout Disabled */
      TIMEOUT_50US     = TWI_TIMEOUT_50US_gc,      /* 50 Microseconds */
      TIMEOUT_100US    = TWI_TIMEOUT_100US_gc,     /* 100 Microseconds */
      TIMEOUT_200US    = TWI_TIMEOUT_200US_gc,     /* 200 Microseconds */
    } TIMEOUT;

    /* Error Code */
    typedef enum
    {
      ERROR_NO_SLAVE = -1,
      ERROR_TRANSFER = -2,
    } ERROR;

    Twi(TWI_t& twi, SDAHOLD hold = SDAHOLD_OFF, SDASETUP setup = SDASETUP_4CYC)
      : _twi(&twi), _ch((&twi - &TWI0) / sizeof(twi))
    {
      _instances[_ch] = this;
      twi.CTRLA   = setup | hold;
      twi.DBGCTRL = 0;
    }

    /* virtual */ ~Twi(void)
    {
      end();
      _instances[_ch] = 0;
    }

    void begin(uint32_t speed = 100000UL, TIMEOUT timeout = TIMEOUT_DISABLED)
    {
      end();
      _twi->CTRLA   = (_twi->CTRLA & ~TWI_FMPEN_bm) | (speed >= 700000UL ? TWI_FMPEN_bm : 0);
      uint16_t baud = Clock::frequency() / speed / 2;
      _twi->MBAUD   = baud <= 7 ? 1 : (baud > 0xFF + 7 ? 0xFF : baud - 7);
      _twi->MSTATUS = ~TWI_BUSSTATE_gm | TWI_BUSSTATE_IDLE_gc;
      _twi->MCTRLB  = 0;
      _twi->MCTRLA  = timeout | TWI_ENABLE_bm;
    }

    void end(void)
    {
      _twi->MCTRLA  = 0;
    }

    void dbgctrl(bool dbgrun = true)
    {
      _twi->DBGCTRL = dbgrun;
    }

    int write(uint16_t addr, const void* buf, uint8_t len, bool stop = true)
    {
      int cnt = ERROR_NO_SLAVE;
      STATE state = start(addr, 0);
      if (state == STATE_ERROR)
        cnt = ERROR_TRANSFER;
      else if (state == STATE_ACK)
      {
        const uint8_t* p = (const uint8_t*)buf;
        for (cnt = 0; cnt < len; ++cnt)
        {
          _twi->MDATA = *p++;
          STATE state = wait();
          if (state == STATE_NAK)
            break;
          if (state == STATE_ERROR)
          {
            cnt = ERROR_TRANSFER;
            break;
          }
        }
        cmd(stop || (state != STATE_ACK) ? CMD_STOP : CMD_REPSTART);
      }
      return cnt;
    }

    int read(uint16_t addr, void* buf, uint8_t len, bool stop = true)
    {
      int cnt = ERROR_NO_SLAVE;
      STATE state = start(addr, 1);
      if (state == STATE_ERROR)
        cnt = ERROR_TRANSFER;
      else if (state == STATE_ACK)
      {
        uint8_t* p = (uint8_t*)buf;
        for (cnt = 0; cnt < len; ++cnt)
        {
          if (cnt)
            cmd(CMD_RECVTRANS);
          STATE state = wait();
          if (state == STATE_ERROR)
          {
            cnt = ERROR_TRANSFER;
            break;
          }
          *p++ = _twi->MDATA;
        }
        cmd(stop || (state == STATE_ERROR) ? CMD_STOP : CMD_REPSTART, ACKACT_NACK);
      }
      return cnt;
    }

  protected:

    /* Acknowledge Action select */
    typedef enum
    {
      ACKACT_ACK  = TWI_ACKACT_ACK_gc,   /* Send ACK */
      ACKACT_NACK = TWI_ACKACT_NACK_gc,  /* Send NACK */
    } ACKACT;

    /* Command select */
    typedef enum
    {
      CMD_NOACT     = TWI_MCMD_NOACT_gc,      /* No Action */
      CMD_REPSTART  = TWI_MCMD_REPSTART_gc,   /* Issue Repeated Start Condition */
      CMD_RECVTRANS = TWI_MCMD_RECVTRANS_gc,  /* Receive or Transmit Data, depending on DIR */
      CMD_STOP      = TWI_MCMD_STOP_gc,       /* Issue Stop Condition */
    } CMD;

    typedef enum
    {
      STATE_ACK,
      STATE_NAK,
      STATE_ERROR,
    } STATE;

    static Twi* _instances[TWI_MAX_CH];

    TWI_t*  _twi;
    uint8_t _ch;

    void cmd(CMD cmd, ACKACT ack = ACKACT_ACK)
    {
      _twi->MCTRLB = (_twi->MCTRLB & ~(TWI_ACKACT_bm | TWI_MCMD_gm)) | cmd | ack;
    }

    STATE start(uint16_t addr, bool read)
    {
      STATE state;
      if (addr & ~0x03FF)
        return STATE_NAK;
      cmd(CMD_NOACT);
      if (addr <= 0x7F)
      {
        _twi->MADDR = (addr << 1) | read; // write or read mode
        state = wait();
      }
      else
      {
        _twi->MADDR = 0xF0 | ((addr >> 8) << 1) | 0; // write mode
        state = wait();
        if (state == STATE_ACK)
        {
          _twi->MDATA = (uint8_t)addr;
          state = wait();
          if ((state == STATE_ACK) && read)
          {
            cmd(CMD_REPSTART);
            _twi->MADDR |= 1; // read mode
            state = wait();
          }
        }
      }
      if (state != STATE_ACK)
        cmd(CMD_STOP);
      return state;
    }

    STATE wait(void)
    {
#if CONFIG_RTC_USE
      uint32_t t = Rtc::read();
      while (Rtc::read() - t < Rtc::us2tick(CONFIG_TWI_BUS_TIMEOUT))
#else
      while (1)
#endif
      {
        uint8_t status = _twi->MSTATUS;
        if (status & (TWI_BUSERR_bm | TWI_ARBLOST_bm))
          break;
        if (status & (TWI_WIF_bm | TWI_RIF_bm))
          return status & TWI_RXACK_bm ? STATE_NAK : STATE_ACK;
        yield();
      }
      return STATE_ERROR;
    }
};

/* Arduino compatible class */
class TwoWire : Twi
{
  public:

    TwoWire(TWI_t& twi, SDAHOLD hold = SDAHOLD_OFF, SDASETUP setup = SDASETUP_4CYC)
      : Twi(twi, hold, setup), _addr(0), _error(0), _count(0), _index(0)
    {
    }
  
    void begin(uint32_t speed = 100000UL, TIMEOUT timeout = TIMEOUT_DISABLED)
    {
      Twi::begin(speed, timeout);
    }
  
    void end(void)
    {
      Twi::end();
    }
  
    void beginTransmission(uint16_t address)
    {
      _addr  = address;
      _count = 0;
      _index = 0;
      _error = 0;
    }

    size_t write(uint8_t data)
    {
      if (_index >= sizeof(_buffer))
      {
        _error = 1;
        return 0;
      }
      _buffer[_index++] = data;
      return 1;
    }

    size_t write(const void *buf, size_t len)
    {
      uint8_t cnt = 0;
      const uint8_t *p = (const uint8_t *)buf;
      while (len--)
        cnt += write(*p++);
      return cnt;
    }

    int endTransmission(bool sendStop = true)
    {
      if (_error)
        return 1; // buffer overflow
      int len = Twi::write(_addr, _buffer, _index, sendStop);
      return len == (int)_index ? 0 : (len < 0 ? 2 : 3);
    }
  
    int requestFrom(uint16_t address, size_t quantity, bool sendStop = true)
    {
      if (quantity > sizeof(_buffer))
        quantity = sizeof(_buffer);
      int len = Twi::read(address, _buffer, quantity, sendStop);
      _index = 0;
      return _count = len > 0 ? len : 0;
    }

    size_t available(void)
    {
      return _index < _count ? _count - _index : 0;
    }

    int read(void)
    {
      return _index < _count ? _buffer[_index++] : -1;
    }

    size_t readBytes(void *buf, size_t len)
    {
      size_t cnt = 0;
      uint8_t* p = (uint8_t*)buf;
      for (; cnt < len; ++cnt)
      {
        int c = read();
        if (c < 0)
          break;
        *p++ = c;
      }
      return cnt;
    }

    int peek(void)
    {
      return _index < _count ? _buffer[_index] : -1;
    }

  private:

    uint16_t _addr;
    int      _error;
    size_t   _count;
    size_t   _index;
    uint8_t  _buffer[CONFIG_TWI_IOBUF_SIZE];
};
