/*
  avr8_usart.h - USART Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include <stddef.h>
#include "avr8_config.h"
#include "avr8_clock.h"

#define USART_ASYNC_BAUDRATE(fcpu, baud) (-((fcpu) * 4 / (baud)))

#if   defined(USART5)
#define USART_MAX_CH 6
#elif defined(USART4)
#define USART_MAX_CH 5
#elif defined(USART3)
#define USART_MAX_CH 4
#elif defined(USART2)
#define USART_MAX_CH 3
#elif defined(USART1)
#define USART_MAX_CH 2
#elif defined(USART0)
#define USART_MAX_CH 1
#else
#define USART_MAX_CH 0
#endif

class Usart
{
  public:

    /* Character Size select */
    typedef enum
    {
      CHSIZE_5BIT  = USART_CHSIZE_5BIT_gc,   /* Character size: 5 bit */
      CHSIZE_6BIT  = USART_CHSIZE_6BIT_gc,   /* Character size: 6 bit */
      CHSIZE_7BIT  = USART_CHSIZE_7BIT_gc,   /* Character size: 7 bit */
      CHSIZE_8BIT  = USART_CHSIZE_8BIT_gc,   /* Character size: 8 bit */
    } CHSIZE;

    /* Parity Mode select */
    typedef enum
    {
      PMODE_DISABLED = USART_PMODE_DISABLED_gc,  /* No Parity */
      PMODE_EVEN     = USART_PMODE_EVEN_gc,      /* Even Parity */
      PMODE_ODD      = USART_PMODE_ODD_gc,       /* Odd Parity */
    } PMODE;

    /* Stop Bit Mode select */
    typedef enum
    {
      SBMODE_1BIT = USART_SBMODE_1BIT_gc,  /* 1 stop bit */
      SBMODE_2BIT = USART_SBMODE_2BIT_gc,  /* 2 stop bits */
    } SBMODE;

    /* USART.STATUS  bit masks and bit positions */
    typedef enum
    {
      STATUS_WFB   = USART_WFB_bm,    /* Wait For Break bit mask. */
      STATUS_BDF   = USART_BDF_bm,    /* Break Detected Flag bit mask. */
      STATUS_IFSIF = USART_ISFIF_bm,  /* Inconsistent Sync Field Interrupt Flag bit mask. */
      STATUS_RXSIF = USART_RXSIF_bm,  /* Receive Start Interrupt bit mask. */
      STATUS_DREIF = USART_DREIF_bm,  /* Data Register Empty Flag bit mask. */
      STATUS_TXCIF = USART_TXCIF_bm,  /* Transmit Interrupt Flag bit mask. */
      STATUS_RXCIF = USART_RXCIF_bm,  /* Receive Complete Interrupt Flag bit mask. */
    } STATUS;

    Usart(USART_t& usart) : _usart(&usart), _ch((&usart - &USART0) / sizeof(usart)), _lbcnt(0), _rxdata(0)
    {
      _instances[_ch] = this;
    }

    /* virtual */ ~Usart(void)
    {
      end();
      _instances[_ch] = 0;
    }

    void begin(int32_t speed, CHSIZE data = CHSIZE_8BIT, PMODE parity = PMODE_DISABLED, SBMODE stop = SBMODE_1BIT, bool onewire = false)
    {
      end();
      _usart->RXPLCTRL = 0;
      _usart->TXPLCTRL = 0;
      _usart->EVCTRL   = 0;
      _usart->DBGCTRL  = 0;
      _usart->BAUD     = speed < 0 ? (uint16_t)-speed : baudrate(speed);
      _usart->CTRLC    = USART_CMODE_ASYNCHRONOUS_gc | parity | stop | data;
      _usart->CTRLB    = (onewire ? USART_ODME_bm : 0) | USART_RXMODE_NORMAL_gc | USART_RXEN_bm | USART_TXEN_bm;
      _usart->STATUS   = USART_TXCIF_bm | USART_RXSIF_bm | USART_ISFIF_bm | USART_BDF_bm;
#if CONFIG_USART_ISR && CONFIG_USART_RXBUF_SIZE
      _usart->CTRLA    = (onewire ? USART_LBME_bm : 0) | USART_RXCIE_bm;
#else
      _usart->CTRLA    = (onewire ? USART_LBME_bm : 0);
#endif
    }

    void end(void)
    {
      _usart->CTRLB = 0;
    }

#if CONFIG_USART_ISR && CONFIG_USART_RXBUF_SIZE
    void runstdby(bool enable = true)
    {
      _usart->CTRLB = (_usart->CTRLB & ~USART_SFDEN_bm) | (enable ? USART_SFDEN_bm : 0);
      _usart->CTRLA = (_usart->CTRLA & ~USART_RXSIE_bm) | (enable ? USART_RXSIE_bm : 0);
    }
#endif

    void clear(void)
    {
      while (read() != -1)
        continue;
    }

    uint8_t available(void)
    {
      return available0();
    }

    int read(void)
    {
      return available() ? read0() : -1;
    }

    uint8_t read(void* buf, uint8_t len)
    {
      uint8_t* p = (uint8_t *)buf;
      uint8_t cnt = 0;
      for (; cnt < len; ++cnt)
      {
        int c = read();
        if (c < 0)
          break;
        *p++ = (uint8_t)c;
      }
      return cnt;      
    }

    uint8_t availableForWrite(void)
    {
      return availableForWrite0();
    }

    void write(uint8_t c)
    {
      while (availableForWrite() == 0)
        yield();
      write0(c);
    }

    void write(const void* buf, uint8_t len)
    {
      const uint8_t* p = (const uint8_t*)buf;
      while (len--)
        write(*p++);
    }

    STATUS status(void)
    {
      return (STATUS)_usart->STATUS;
    }      

    void dbgctrl(bool dbgrun = true)
    {
      _usart->DBGCTRL = dbgrun ? USART_DBGRUN_bm : 0;
    }

    static void poll(void)
    {
#if !CONFIG_USART_ISR && (CONFIG_USART_TXBUF_SIZE || CONFIG_USART_RXBUF_SIZE)
      for (uint8_t i = 0; i < USART_MAX_CH; ++i)
      {
        Usart* usart = _instances[i];
        if (usart)
        {
#if CONFIG_USART_TXBUF_SIZE
          usart->isr_dre();
#endif
#if CONFIG_USART_RXBUF_SIZE
          usart->isr_rxc();
#endif
        }
      }
#endif
    }

  protected:

    static Usart *_instances[USART_MAX_CH];

    USART_t*         _usart;
    uint8_t          _ch;
    volatile uint8_t _lbcnt;
    uint8_t          _rxdata;

    uint16_t baudrate(uint32_t speed, uint16_t vdd = CONFIG_VDD)
    {
      uint8_t mask = 0;
      uint8_t samples = 1;
      switch (_usart->CTRLC & USART_CMODE_gm)
      {
        case USART_CMODE_ASYNCHRONOUS_gc: samples = (_usart->CTRLB & USART_RXMODE_gm) == USART_RXMODE_CLK2X_gc ? 64 / 8 : 64 / 16; break;
        case USART_CMODE_SYNCHRONOUS_gc:  samples = 64 /  2; mask = 0x3F; break;
        case USART_CMODE_IRCOM_gc:        samples = 64 / 16; break;
        case USART_CMODE_MSPI_gc:         samples = 64 /  1; mask = 0x3F; break;
      }
#if defined(SIGROW_OSC20ERR3V)
      return (Clock::frequency() * samples / speed * (Sigrow::oscerr(vdd) + 1024) / 1024) & ~mask;
#else
      return (Clock::frequency() * samples / speed) & ~mask;
#endif
    }

    void txdata(uint8_t c)
    {
      uint8_t save = SREG;
      cli();
      if (_usart->CTRLA & USART_LBME_bm)
        ++_lbcnt;
      _usart->TXDATAL = c;
      SREG = save;
    }

    int rxdata(void)
    {
      uint8_t save = SREG;
      cli();
      int c = _usart->RXDATAL;
      if (_lbcnt)
      {
        --_lbcnt;
        c = -1;
      }
      SREG = save;
      return c;
    }

#if CONFIG_USART_RXBUF_SIZE

    volatile uint8_t _rxrdpos;
    volatile uint8_t _rxwrpos;
    uint8_t _rxbuffer[CONFIG_USART_RXBUF_SIZE];

#if CONFIG_USART_ISR
    friend void usart_isr_rxc(uint8_t ch);
#endif

    inline void isr_rxc(void)
    {
      uint8_t flags = _usart->STATUS;
      _usart->STATUS = (flags & USART_RXSIF_bm);
      if (flags & USART_RXCIF_bm)
      {
        int c = rxdata();
        if (c < 0)
          ; /* loop-back mode */
        else if (!full())
          _rxbuffer[rxpos(_rxwrpos++)] = (uint8_t)c;
        else
          ; /* buffer overflow */
      }
    }

    inline bool full(void)
    {
      return ((_rxwrpos - _rxrdpos) & (CONFIG_USART_RXBUF_SIZE | (CONFIG_USART_RXBUF_SIZE - 1))) == CONFIG_USART_RXBUF_SIZE;
    }

    inline static uint8_t rxpos(uint8_t pos)
    {
      return pos & (CONFIG_USART_RXBUF_SIZE - 1);
    }

    inline uint8_t available0(void)
    {
#if !CONFIG_USART_ISR
      isr_rxc();
#endif
      int len = (int)rxpos(_rxwrpos) - rxpos(_rxrdpos);
      return len == 0 ? (full() ? CONFIG_USART_RXBUF_SIZE : 0) : (len > 0 ? len : len + CONFIG_USART_RXBUF_SIZE);
    }

    inline int read0(void)
    {
      return _rxbuffer[rxpos(_rxrdpos++)];
    }

#else

    inline uint8_t available0(void)
    {
      if (_usart->STATUS & USART_RXCIE_bm)
      {
        int c = rxdata();
        if (c >= 0)
        {
          _rxdata = (uint8_t)c;
          return 1;
        }          
      }
      return 0;
    }

    inline uint8_t read0(void)
    {
      return _rxdata;
    }

#endif

#if CONFIG_USART_TXBUF_SIZE

    volatile uint8_t _txrdpos;
    volatile uint8_t _txwrpos;
    uint8_t _txbuffer[CONFIG_USART_TXBUF_SIZE];

#if CONFIG_USART_ISR
    friend void usart_isr_dre(uint8_t ch);
#endif

    inline void isr_dre(void)
    {
      if (_usart->STATUS & USART_DREIF_bm)
      {
        if (!empty())
          txdata(_txbuffer[txpos(_txrdpos++)]);
        else /* buffer empty */;
          _usart->CTRLA &= ~USART_DREIE_bm;
      }
    }

    inline bool empty(void)
    {
      return ((_txwrpos - _txrdpos) & (CONFIG_USART_TXBUF_SIZE | (CONFIG_USART_TXBUF_SIZE - 1))) == 0;
    }

    inline static uint8_t txpos(uint8_t pos)
    {
      return pos & (CONFIG_USART_TXBUF_SIZE - 1);
    }

    inline uint8_t availableForWrite0(void)
    {
#if !CONFIG_USART_ISR
      isr_dre();
#endif
      int len = (int)txpos(_txrdpos) - txpos(_txwrpos);
      return len == 0 ? (empty() ? CONFIG_USART_TXBUF_SIZE : 0) : (len > 0 ? len : len + CONFIG_USART_TXBUF_SIZE);
    }

    inline void write0(uint8_t c)
    {
      _txbuffer[txpos(_txwrpos++)] = c;
#if CONFIG_USART_ISR
      _usart->CTRLA |= USART_DREIE_bm;
#endif
    }

#else

    inline uint8_t availableForWrite0(void)
    {
      return _usart->STATUS & USART_DREIE_bm ? 1 : 0;
    }

    inline void write0(uint8_t c)
    {
      txdata(c);
    }

#endif
};
