/*
  avr8_vref.h - VREF Driver for Microchip AVR8 Series
 
  Copyright (c) 2025 Sasapea's Lab. All right reserved.
 
  This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of
  the License, or at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <https://www.gnu.org/licenses/>.
*/
#pragma once

#include "avr8_config.h"

class Vref
{
  public:

#if defined(__AVR_0__)

    typedef enum
    {
      ACREF_0V55  = VREF_DAC0REFSEL_0V55_gc,  /* Voltage reference at 0.55V */
      ACREF_1V1   = VREF_DAC0REFSEL_1V1_gc,   /* Voltage reference at 1.1V */
      ACREF_2V5   = VREF_DAC0REFSEL_2V5_gc,   /* Voltage reference at 2.5V */
      ACREF_4V34  = VREF_DAC0REFSEL_4V34_gc,  /* Voltage reference at 4.34V */
      ACREF_1V5   = VREF_DAC0REFSEL_1V5_gc,   /* Voltage reference at 1.5V */
    } ACREF;

    typedef enum
    {
      ADCREF_0V55 = VREF_ADC0REFSEL_0V55_gc,  /* Voltage reference at 0.55V */
      ADCREF_1V1  = VREF_ADC0REFSEL_1V1_gc,   /* Voltage reference at 1.1V */
      ADCREF_2V5  = VREF_ADC0REFSEL_2V5_gc,   /* Voltage reference at 2.5V */
      ADCREF_4V34 = VREF_ADC0REFSEL_4V34_gc,  /* Voltage reference at 4.34V */
      ADCREF_1V5  = VREF_ADC0REFSEL_1V5_gc,   /* Voltage reference at 1.5V */
      ADCREF_VDD  = 7 << VREF_ADC0REFSEL_gp,
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      (void)ch;
      VREF.CTRLA = (VREF.CTRLA & ~VREF_DAC0REFSEL_gm) | ref;
      VREF.CTRLB = (VREF.CTRLB & ~VREF_DAC0REFEN_bm) | (enable ? VREF_DAC0REFEN_bm : 0);
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      (void)vdd;
      (void)ref;
      return vrefmv((VREF.CTRLA & VREF_DAC0REFSEL_gm) >> VREF_DAC0REFSEL_gp);
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      (void)ch;
      if (ref == ADCREF_VDD)
        ADC0.CTRLC = (ADC0.CTRLC & ~ADC_REFSEL_gm) | ADC_SAMPCAP_bm | ADC_REFSEL_VDDREF_gc;
      else
      {
        ADC0.CTRLC = (ADC0.CTRLC & ~(ADC_SAMPCAP_bm | ADC_REFSEL_gm)) | (ref != ADCREF_0V55 ? ADC_SAMPCAP_bm : 0) | ADC_REFSEL_INTREF_gc;
        VREF.CTRLA = (VREF.CTRLA & ~VREF_ADC0REFSEL_gm) | ref;
        VREF.CTRLB = (VREF.CTRLB & ~VREF_ADC0REFEN_bm) | (enable ? VREF_ADC0REFEN_bm : 0);
      }
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      (void)ref;
      return (ADC0.CTRLA & ADC_REFSEL_gm) == ADC_REFSEL_VDDREF_gc
        ? vdd : vrefmv((VREF.CTRLA & VREF_ADC0REFSEL_gm) >> VREF_ADC0REFSEL_gp);
    }

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 550, 1100, 2500, 4340, 1500, 0, 0, 0 };
      return VOLT[index];
    }

#elif defined(__AVR_1__)

    typedef enum
    {
      ACREF_0V55  = VREF_DAC0REFSEL_0V55_gc,  /* Voltage reference at 0.55V */
      ACREF_1V1   = VREF_DAC0REFSEL_1V1_gc,   /* Voltage reference at 1.1V */
      ACREF_2V5   = VREF_DAC0REFSEL_2V5_gc,   /* Voltage reference at 2.5V */
      ACREF_4V34  = VREF_DAC0REFSEL_4V34_gc,  /* Voltage reference at 4.34V */
      ACREF_1V5   = VREF_DAC0REFSEL_1V5_gc,   /* Voltage reference at 1.5V */
    } ACREF;

    typedef enum
    {
      DACREF_0V55 = VREF_DAC0REFSEL_0V55_gc,  /* Voltage reference at 0.55V */
      DACREF_1V1  = VREF_DAC0REFSEL_1V1_gc,   /* Voltage reference at 1.1V */
      DACREF_2V5  = VREF_DAC0REFSEL_2V5_gc,   /* Voltage reference at 2.5V */
      DACREF_4V34 = VREF_DAC0REFSEL_4V34_gc,  /* Voltage reference at 4.34V */
      DACREF_1V5  = VREF_DAC0REFSEL_1V5_gc,   /* Voltage reference at 1.5V */
    } DACREF;

    typedef enum
    {
      ADCREF_0V55 = VREF_ADC0REFSEL_0V55_gc,  /* Voltage reference at 0.55V */
      ADCREF_1V1  = VREF_ADC0REFSEL_1V1_gc,   /* Voltage reference at 1.1V */
      ADCREF_2V5  = VREF_ADC0REFSEL_2V5_gc,   /* Voltage reference at 2.5V */
      ADCREF_4V34 = VREF_ADC0REFSEL_4V34_gc,  /* Voltage reference at 4.34V */
      ADCREF_1V5  = VREF_ADC0REFSEL_1V5_gc,   /* Voltage reference at 1.5V */
      ADCREF_VDD  = 7 << VREF_ADC0REFSEL_gp,
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      dac(ch, (DACREF)ref, enable);
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      return dac(ch, vdd, ref);
    }

    static void dac(uint8_t ch, DACREF ref, bool enable = false)
    {
      switch (ch)
      {
        case 0:
          VREF.CTRLA = (VREF.CTRLA & ~VREF_DAC0REFSEL_gm) | ref;
          VREF.CTRLB = (VREF.CTRLB & ~VREF_DAC0REFEN_bm) | (enable ? VREF_DAC0REFEN_bm : 0);
          break;
#if defined(VREF_DAC1REFSEL_gm)
        case 1:
          VREF.CTRLC = (VREF.CTRLC & ~VREF_DAC1REFSEL_gm) | ref;
          VREF.CTRLB = (VREF.CTRLB & ~VREF_DAC1REFEN_bm) | (enable ? VREF_DAC1REFEN_bm : 0);
          break;
#endif
#if defined(VREF_DAC2REFSEL_gm)
        case 2:
          VREF.CTRLD = (VREF.CTRLD & ~VREF_DAC2REFSEL_gm) | ref;
          VREF.CTRLB = (VREF.CTRLB & ~VREF_DAC2REFEN_bm) | (enable ? VREF_DAC2REFEN_bm : 0);
          break;
#endif
      }
    }

    static uint16_t dac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)vdd;
      (void)ref;
      switch (ch)
      {
        case 0: return vrefmv((VREF.CTRLA & VREF_DAC0REFSEL_gm) >> VREF_DAC0REFSEL_gp);
#if defined(VREF_DAC1REFSEL_gm)
        case 1: return vrefmv((VREF.CTRLC & VREF_DAC1REFSEL_gm) >> VREF_DAC1REFSEL_gp);
#endif
#if defined(VREF_DAC2REFSEL_gm)
        case 2: return vrefmv((VREF.CTRLD & VREF_DAC2REFSEL_gm) >> VREF_DAC2REFSEL_gp);
#endif
      }
      return 0;
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      switch (ch)
      {
        case 0:
          if (ref == ADCREF_VDD)
            ADC0.CTRLC = (ADC0.CTRLC & ~ADC_REFSEL_gm) | ADC_SAMPCAP_bm | ADC_REFSEL_VDDREF_gc;
          else
          {
            ADC0.CTRLC = (ADC0.CTRLC & ~(ADC_SAMPCAP_bm | ADC_REFSEL_gm)) | (ref != ADCREF_0V55 ? ADC_SAMPCAP_bm : 0) | ADC_REFSEL_INTREF_gc;
            VREF.CTRLA = (VREF.CTRLA & ~VREF_ADC0REFSEL_gm) | ref;
            VREF.CTRLB = (VREF.CTRLB & ~VREF_ADC0REFEN_bm) | (enable ? VREF_ADC0REFEN_bm : 0);
          }            
          break;
#if defined(VREF_ADC1REFSEL_gm)
        case 1:
          if (ref == ADCREF_VDD)
            ADC1.CTRLC = (ADC1.CTRLC & ~ADC_REFSEL_gm) | ADC_SAMPCAP_bm | ADC_REFSEL_VDDREF_gc;
          else
          {
            ADC1.CTRLC = (ADC1.CTRLC & ~(ADC_SAMPCAP_bm | ADC_REFSEL_gm)) | (ref != ADCREF_0V55 ? ADC_SAMPCAP_bm : 0) | ADC_REFSEL_INTREF_gc;
            VREF.CTRLC = (VREF.CTRLC & ~VREF_ADC1REFSEL_gm) | ref;
            VREF.CTRLB = (VREF.CTRLB & ~VREF_ADC1REFEN_bm) | (enable ? VREF_ADC1REFEN_bm : 0);
          }            
          break;
#endif
      }
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)vdd;
      (void)ref;
      switch (ch)
      {
        case 0:
          return (ADC0.CTRLA & ADC_REFSEL_gm) == ADC_REFSEL_VDDREF_gc
            ? vdd : vrefmv((VREF.CTRLA & VREF_ADC0REFSEL_gm) >> VREF_ADC0REFSEL_gp);
#if defined(VREF_ADC1REFSEL_gm)
        case 1:
          return (ADC1.CTRLA & ADC_REFSEL_gm) == ADC_REFSEL_VDDREF_gc
            ? vdd : vrefmv((VREF.CTRLC & VREF_ADC1REFSEL_gm) >> VREF_ADC1REFSEL_gp);
#endif
      }
      return 0;
    }

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 550, 1100, 2500, 4340, 1500, 0, 0, 0 };
      return VOLT[index];
    }

#elif defined(__AVR_2__)

    typedef enum
    {
      ACREF_1V024  = VREF_AC0REFSEL_1V024_gc,  /* Internal 1.024V Reference */
      ACREF_2V048  = VREF_AC0REFSEL_2V048_gc,  /* Internal 2.048V Reference */
      ACREF_2V500  = VREF_AC0REFSEL_2V5_gc,    /* Internal 2.5V Reference */
      ACREF_4V096  = VREF_AC0REFSEL_4V096_gc,  /* Internal 4.096V Reference */
      ACREF_VDD    = VREF_AC0REFSEL_AVDD_gc,   /* VDD */
    } ACREF;

    typedef enum
    {
      ADCREF_VDD   = ADC_REFSEL_VDD_gc,       /* VDD as reference */
      ADCREF_VREFA = ADC_REFSEL_VREFA_gc,     /* External reference on VREFA pin */
      ADCREF_1V024 = ADC_REFSEL_1024MV_gc,    /* Internal 1.024V reference */
      ADCREF_2V048 = ADC_REFSEL_2048MV_gc,    /* Internal 2.048V reference */
      ADCREF_2V500 = ADC_REFSEL_2500MV_gc,    /* Internal 2.500V reference */
      ADCREF_4V096 = ADC_REFSEL_4096MV_gc,    /* Internal 4.096V reference */
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      (void)ch;
      VREF.CTRLA = (VREF.CTRLA & ~VREF_AC0REFSEL_gm) | ref;
      VREF.CTRLB = (VREF.CTRLB & ~VREF_AC0REFEN_bm) | (enable ? VREF_AC0REFEN_bm : 0);
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      (void)ref;
      uint8_t sel = VREF.CTRLA & VREF_AC0REFSEL_gm;
      switch (sel)
      {
        case VREF_AC0REFSEL_AVDD_gc: return vdd;
      }
      return vrefmv(sel >> VREF_AC0REFSEL_gp);
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      (void)ch;
      ADC0.CTRLC = (ADC0.CTRLC & ~ADC_REFSEL_gm) | ref;
      VREF.CTRLB = (VREF.CTRLB & ~VREF_ADC0REFEN_bm) | (enable ? VREF_ADC0REFEN_bm : 0);
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = ADC0.CTRLA & ADC_REFSEL_gm;
      switch (sel)
      {
        case ADC_REFSEL_VDD_gc  : return vdd;
        case ADC_REFSEL_VREFA_gc: return ref;
      }
      return adcmv(sel >> ADC_REFSEL_gp);
    }

    static void nvm(bool enable = true)
    {
      VREF.CTRLB = (VREF.CTRLB & ~VREF_NVMREFEN_bm) | (enable ? VREF_NVMREFEN_bm : 0);
    }      

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 1024, 2048, 2500, 4096, 0, 0, 0, 0 };
      return VOLT[index];
    }

    static uint16_t adcmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 0, 0, 0, 0, 1024, 2048, 2500, 4096 };
      return VOLT[index];
    }

#elif defined(__AVR_DA__) || defined(__AVR_DB__) || defined(__AVR_DD__)

    typedef enum
    {
      ACREF_1V024 = VREF_REFSEL_1V024_gc,     /* Internal 1.024V reference */
      ACREF_2V048 = VREF_REFSEL_2V048_gc,     /* Internal 2.048V reference */
      ACREF_4V096 = VREF_REFSEL_4V096_gc,     /* Internal 4.096V reference */
      ACREF_2V500 = VREF_REFSEL_2V500_gc,     /* Internal 2.500V reference */
      ACREF_VDD   = VREF_REFSEL_VDD_gc,       /* VDD as reference */
      ACREF_VREFA = VREF_REFSEL_VREFA_gc,     /* External reference on VREFA pin */
    } ACREF;

    typedef enum
    {
      DACREF_1V024 = VREF_REFSEL_1V024_gc,    /* Internal 1.024V reference */
      DACREF_2V048 = VREF_REFSEL_2V048_gc,    /* Internal 2.048V reference */
      DACREF_4V096 = VREF_REFSEL_4V096_gc,    /* Internal 4.096V reference */
      DACREF_2V500 = VREF_REFSEL_2V500_gc,    /* Internal 2.500V reference */
      DACREF_VDD   = VREF_REFSEL_VDD_gc,      /* VDD as reference */
      DACREF_VREFA = VREF_REFSEL_VREFA_gc,    /* External reference on VREFA pin */
    } DACREF;

    typedef enum
    {
      ADCREF_1V024 = VREF_REFSEL_1V024_gc,    /* Internal 1.024V reference */
      ADCREF_2V048 = VREF_REFSEL_2V048_gc,    /* Internal 2.048V reference */
      ADCREF_4V096 = VREF_REFSEL_4V096_gc,    /* Internal 4.096V reference */
      ADCREF_2V500 = VREF_REFSEL_2V500_gc,    /* Internal 2.500V reference */
      ADCREF_VDD   = VREF_REFSEL_VDD_gc,      /* VDD as reference */
      ADCREF_VREFA = VREF_REFSEL_VREFA_gc,    /* External reference on VREFA pin */
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      /* AC0 - AC2 */
      (void)ch;
      VREF.ACREF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.ACREF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

    static void dac(uint8_t ch, DACREF ref, bool enable = false)
    {
      (void)ch;
      VREF.DAC0REF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t dac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.DAC0REF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      (void)ch;
      VREF.ADC0REF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.ADC0REF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 1024, 2048, 2500, 4096, 0, 0, 0, 0 };
      return VOLT[index];
    }

#elif defined(__AVR_EA__)

    typedef enum
    {
      ACREF_1V024 = VREF_REFSEL_1V024_gc,     /* Internal 1.024V reference */
      ACREF_2V048 = VREF_REFSEL_2V048_gc,     /* Internal 2.048V reference */
      ACREF_4V096 = VREF_REFSEL_4V096_gc,     /* Internal 4.096V reference */
      ACREF_2V500 = VREF_REFSEL_2V500_gc,     /* Internal 2.500V reference */
      ACREF_VDD   = VREF_REFSEL_VDD_gc,       /* VDD as reference */
      ACREF_VREFA = VREF_REFSEL_VREFA_gc,     /* External reference on VREFA pin */
    } ACREF;

    typedef enum
    {
      DACREF_1V024 = VREF_REFSEL_1V024_gc,    /* Internal 1.024V reference */
      DACREF_2V048 = VREF_REFSEL_2V048_gc,    /* Internal 2.048V reference */
      DACREF_4V096 = VREF_REFSEL_4V096_gc,    /* Internal 4.096V reference */
      DACREF_2V500 = VREF_REFSEL_2V500_gc,    /* Internal 2.500V reference */
      DACREF_VDD   = VREF_REFSEL_VDD_gc,      /* VDD as reference */
      DACREF_VREFA = VREF_REFSEL_VREFA_gc,    /* External reference on VREFA pin */
    } DACREF;

    typedef enum
    {
      ADCREF_VDD   = ADC_REFSEL_VDD_gc,      /* VDD as reference */
      ADCREF_VREFA = ADC_REFSEL_VREFA_gc,    /* External reference on VREFA pin */
      ADCREF_1V024 = ADC_REFSEL_1V024_gc,    /* Internal 1.024V reference */
      ADCREF_2V048 = ADC_REFSEL_2V048_gc,    /* Internal 2.048V reference */
      ADCREF_4V096 = ADC_REFSEL_4V096_gc,    /* Internal 4.096V reference */
      ADCREF_2V500 = ADC_REFSEL_2V500_gc,    /* Internal 2.500V reference */
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      /* AC0 - AC1 */
      (void)ch;
      VREF.ACREF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.ACREF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

    static void dac(uint8_t ch, DACREF ref, bool enable = false)
    {
      (void)ch;
      VREF.DAC0REF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t dac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.DAC0REF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      (void)ch;
      (void)enable;
      ADC0.CTRLC = (ADC0.CTRLC & ~ADC_REFSEL_gm) | ref;
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = ADC0.CTRLC & ADC_REFSEL_gm;
      switch (sel)
      {
        case ADC_REFSEL_VDD_gc  : return vdd;
        case ADC_REFSEL_VREFA_gc: return ref;
      }
      return adcmv(sel >> ADC_REFSEL_gp);
    }

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 1024, 2048, 2500, 4096, 0, 0, 0, 0 };
      return VOLT[index];
    }

    static uint16_t adcmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 0, 0, 0, 0, 1024, 2048, 2500, 4096 };
      return VOLT[index];
    }

#elif defined(__AVR_DU__) || defined(__AVR_EB__)

    typedef enum
    {
      ACREF_1V024 = VREF_REFSEL_1V024_gc,     /* Internal 1.024V reference */
      ACREF_2V048 = VREF_REFSEL_2V048_gc,     /* Internal 2.048V reference */
      ACREF_4V096 = VREF_REFSEL_4V096_gc,     /* Internal 4.096V reference */
      ACREF_2V500 = VREF_REFSEL_2V500_gc,     /* Internal 2.500V reference */
      ACREF_VDD   = VREF_REFSEL_VDD_gc,       /* VDD as reference */
      ACREF_VREFA = VREF_REFSEL_VREFA_gc,     /* External reference on VREFA pin */
    } ACREF;

    typedef enum
    {
      ADCREF_VDD   = ADC_REFSEL_VDD_gc,      /* VDD as reference */
      ADCREF_VREFA = ADC_REFSEL_VREFA_gc,    /* External reference on VREFA pin */
      ADCREF_1V024 = ADC_REFSEL_1V024_gc,    /* Internal 1.024V reference */
      ADCREF_2V048 = ADC_REFSEL_2V048_gc,    /* Internal 2.048V reference */
      ADCREF_4V096 = ADC_REFSEL_4V096_gc,    /* Internal 4.096V reference */
      ADCREF_2V500 = ADC_REFSEL_2V500_gc,    /* Internal 2.500V reference */
    } ADCREF;

    static void ac(uint8_t ch, ACREF ref, bool enable = false)
    {
      /* AC0 - AC1 */
      (void)ch;
      VREF.ACREF = (enable ? VREF_ALWAYSON_bm : 0) | ref;
    }

    static uint16_t ac(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = VREF.ACREF & VREF_REFSEL_gm;
      switch (sel)
      {
        case VREF_REFSEL_VDD_gc  : return vdd;
        case VREF_REFSEL_VREFA_gc: return ref;
      }
      return vrefmv(sel >> VREF_REFSEL_gp);
    }

    static void adc(uint8_t ch, ADCREF ref, bool enable = false)
    {
      (void)ch;
      (void)enable;
      ADC0.CTRLC = (ADC0.CTRLC & ~ADC_REFSEL_gm) | ref;
    }

    static uint16_t adc(uint8_t ch, uint16_t vdd = CONFIG_VDD, uint16_t ref = CONFIG_VREFA)
    {
      (void)ch;
      uint8_t sel = ADC0.CTRLC & ADC_REFSEL_gm;
      switch (sel)
      {
        case ADC_REFSEL_VDD_gc  : return vdd;
        case ADC_REFSEL_VREFA_gc: return ref;
      }
      return adcmv(sel >> ADC_REFSEL_gp);
    }

  protected:

    static uint16_t vrefmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 1024, 2048, 2500, 4096, 0, 0, 0, 0 };
      return VOLT[index];
    }

    static uint16_t adcmv(uint8_t index)
    {
      static const uint16_t VOLT[] = { 0, 0, 0, 0, 1024, 2048, 2500, 4096 };
      return VOLT[index];
    }

#endif
};
